﻿using UnityEngine;
using System;
using System.Collections;

abstract class Note
{
    // define common note properties

    public double Time;
    public HandType HandType;

    protected Level level;
    protected NoteObject noteObject;

    // get only, make this into property?
    public bool Active;

    // interpret note option
    protected abstract void FromBmsNum(String num);
    protected abstract NoteObject CreateNoteObjectImpl();

    // instantiate associated game object

    public void CreateNoteObject()
    {
        noteObject = CreateNoteObjectImpl();
    }

    // override this function to implement per-note judgement.
    public virtual JudgeType JudgeTiming(double time)
    {
        double delta = time - this.Time;
        var timing = Math.Abs(delta);
        var early = delta < 0;

        if (timing < 0.1)
        {
            return JudgeType.Perfect;
        }
        else if (timing < 0.5)
        {
            return JudgeType.Hit;
        } else if (timing < 1.0 && early)
        {
            return JudgeType.Miss;
        }
        else
        {
            return JudgeType.Ignore;
        }
    }

    public virtual HitType CheckHit(Ray ray)
    {
        // add near support

        if(noteObject.IsHit(ray))
        {
            return HitType.Hit;
        } else
        {
            return HitType.Miss;
        }
    }

    public void HandleJudge(JudgeResult judge)
    {
        if (judge.type != JudgeType.Ignore)
        {
            Debug.Log(string.Format("Note #{0}: {1}, Correct Hand: {2}", 
                0,
                System.Enum.GetName(typeof(JudgeType), judge.type),
                judge.correctHand
            ));
            noteObject.NoteHit(judge);
        }
    }


    // TODO: Refactort this
    private static readonly float NOTE_SHOW_TIMING = 5;
    private static readonly float NOTE_HIDE_TIMING = -2;
    private static readonly float NOTE_POSITION_MULTIPLIER = 2;
    public void Update(float remainingTime)
    {
        if (remainingTime < NOTE_HIDE_TIMING || remainingTime > NOTE_SHOW_TIMING)
        {
            noteObject.gameObject.SetActive(false);
            Active = false;
        }
        else
        {
            Active = true;
            noteObject.gameObject.SetActive(true);
        }

        noteObject.SetPosition(remainingTime/NOTE_SHOW_TIMING * NOTE_POSITION_MULTIPLIER);
    }

    // factory method
    // may return null if given note is not supported
    public static Note Create(BmsNote bn, double bpm, Level parent)
    {
        Note note = null;
        HandType handType = HandType.None;

        switch (bn.lane)
        {
            case "11": // 1p first lane: FORWARD LEFT HAND
                note = new ForwardNote();
                handType = HandType.Left;
                break;
            case "12": // 1p second lane: FORWARD RIGHT HAND
                note = new ForwardNote();
                handType = HandType.Right;
                break;
            case "13": // 1p third lane: EDGE LEFT HAND
                note = new EdgeNote();
                handType = HandType.Left;

                break;
            case "14": // 1p forth lane: EDGE RIGHT HAND
                note = new EdgeNote();
                handType = HandType.Right;

                break;
            
            // TODO : add action notes

            default:
                return null;
        }

        note.FromBmsNum(bn.num);
        note.HandType = handType;
        note.Time = bn.barTime * (60.0 / bpm * 4);

        note.level = parent;

        note.CreateNoteObject();

        return note;
    }
}

class ForwardNote : Note
{
    private float x, y;

    protected override NoteObject CreateNoteObjectImpl()
    {
        var obj = MonoBehaviour.Instantiate<ForwardNoteObject>(PlayEngine.inst.ForwardNoteObject);
        obj.Init(x, y, HandType);
        return obj;
    }

    protected override void FromBmsNum(string num)
    {
        //Debug.Log(num);
        // since 00 is used for empty notes, 1 ~ Z is valid range
        x = charToCoord(num[0]) / 35.0f;
        y = charToCoord(num[1]) / 35.0f;
    }

    private int charToCoord(char a)
    {
        if (char.IsDigit(a))
        {
            return a - '1';
        }
        else if (char.IsLetter(a))
        {
            return char.ToLower(a) - 'a' + 9;
        }

        throw new ArgumentException("wrong data");
    }


}

class RearNote : Note
{
    protected override NoteObject CreateNoteObjectImpl()
    {
        throw new NotImplementedException();
    }

    protected override void FromBmsNum(string num)
    {
        throw new NotImplementedException();
    }
}

class EdgeNote : Note
{
    private int index;

    protected override NoteObject CreateNoteObjectImpl()
    {
        var obj = MonoBehaviour.Instantiate<EdgeNoteObject>(PlayEngine.inst.EdgeNoteObjects[index]);
        obj.Init(HandType);
        return obj;
    }

    protected override void FromBmsNum(string num)
    {
        index = charToIndex(num);
    }

    private int charToIndex(string hex)
    {
        if (hex[0] != 'E')
        {
            throw new ArgumentException("wrong data");
        }
        if (char.IsDigit(hex[1]))
        {
            return hex[1] - '0';
        }

        throw new ArgumentException("wrong data");
    }

    private int GetPrefabIndex()
    {
        return 1;
    }
}

public enum NoteType
{
    Front,
    Rear,
    Edge,
    Action
}

public enum HandType
{
    Left,
    Right,
    None
}