﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Windows.Kinect;

namespace MotionAnalysis
{
    public class DistItvExtractor
    {
        private Dictionary<string, CameraSpacePoint?> Points =
            new Dictionary<string, CameraSpacePoint?>
            {
                { "Head",            null },
                { "HandLeft",        null },
                { "HandRight",       null },
                { "SpineShoulder",   null },
                { "SpineMid",        null },
                { "ElbowLeft",       null },
                { "ElbowRight",      null },
                { "KneeLeft",        null },
                { "KneeRight",       null },
                { "SpineMidRecent",  null },
                { "HandLeftRecent",  null },
                { "HandRightRecent", null }
            };
    
        private float            Table;
    
        public float? DistHandBaseSpineShoulder
        { get; private set; }
        public float? DistSpine
        { get; private set; }
        public float? DistHandBaseHead
        { get; private set; }
        public float? DistHandLeft
        { get; private set; }
        public float? DistHandRight
        { get; private set; }
        public float? DistHandBaseElbow_Left
        { get; private set; }
        public float? DistHandBaseElbow_Right
        { get; private set; }
        public float? DistHandBaseHead_Left
        { get; private set; }
        public float? DistHandBaseHead_Right
        { get; private set; }
        public float? DistHandBaseSpineMid_Left
        { get; private set; }
        public float? DistHandBaseSpineMid_Right
        { get; private set; }
        public float? DistKneeBaseTable
        { get; private set; }
        public float? ItvHand
        { get; private set; }
        public float? ItvElbowBaseSpineMid_Left
        { get; private set; }
        public float? ItvElbowBaseSpineMid_Right
        { get; private set; }
        public float? ItvHandBaseHead
        { get; private set; }
        public float? ItvHandBaseHead_LeftEar
        { get; private set; }
        public float? ItvHandBaseHead_RightEar
        { get; private set; }
        public float? ItvHandBaseHead_LeftDepth
        { get; private set; }
        public float? ItvHandBaseHead_RightDepth
        { get; private set; }
    
        public DistItvExtractor(float table)
        {
            Table = table;
        }
    
        public void Extract(KinectModule.IBody body)
        {
            UpdatePosition(body);
            UpdateDistItv();
            SaveRecent();
        }
    
        private void UpdatePosition(KinectModule.IBody body)
        {
            Points.Keys.Select(x => (JointType)Enum.Parse(typeof(JointType), x))
                       .ToList()
                       .ForEach(x => Points[x.ToString()] =
                            body.Joints.ContainsKey(x) ?
                            (CameraSpacePoint?)body.Joints[x].Position : null);
        }
        
        private void SaveRecent()
        {
            Points["SpineMidRecent"]  = Points["SpineMid"];
            Points["HandLeftRecent"]  = Points["HandLeft"];
            Points["HandRightRecent"] = Points["HandRight"];
        }
        
        private void UpdateDistItv()
        {
            ComputeClap();
            ComputeJump();
            ComputeHurray();
            ComputeHandMove();
            ComputeGuardBase();
            ComputeHandUp();
            ComputeHandDown();
            ComputeOnTheTable();
            ComputeJesus();
            ComputeHeadphone();
        }

        bool PointsNotNull(string[] names)
        {
            foreach(var x in names)
                if (Points[x] == null)
                    return false;

            return true;
        }

        private void ComputeClap()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "SpineShoulder",
            }))
            {
                DistHandBaseSpineShoulder = null;

                ItvHand                   = null;

                return;
            }

            var handleft      = Points["HandLeft"]     .Value;
            var handright     = Points["HandRight"]    .Value;
            var spineshoulder = Points["SpineShoulder"].Value;

            DistHandBaseSpineShoulder = Mathf.Min(handleft.Y, handright.Y) - spineshoulder.Y;

            ItvHand                   = Mathf.Sqrt(Mathf.Pow(handleft.X - handright.X, 2.0f) +
                                                   Mathf.Pow(handleft.Y - handright.Y, 2.0f) +
                                                   Mathf.Pow(handleft.Z - handright.Z, 2.0f));
        }

        private void ComputeJump()
        {
            if (!PointsNotNull(new string[]
            {
                "SpineMid",
                "SpineMidRecent"
            }))
            {
                DistSpine = null;

                return;
            }

            var spinemid       = Points["SpineMid"]      .Value;
            var spinemidrecent = Points["SpineMidRecent"].Value;

            DistSpine = spinemid.Y - spinemidrecent.Y;
        }

        private void ComputeHurray()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "Head"
            }))
            {
                DistHandBaseHead = null;

                return;
            }

            var handleft  = Points["HandLeft"] .Value;
            var handright = Points["HandRight"].Value;
            var head      = Points["Head"]     .Value;

            DistHandBaseHead = Mathf.Min(handleft.Y, handright.Y) - head.Y;
        }

        private void ComputeHandMove()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "HandLeftRecent",
                "HandRightRecent"
            }))
            {
                DistHandLeft  = null;
                DistHandRight = null;

                return;
            }

            var handleft        = Points["HandLeft"]       .Value;
            var handright       = Points["HandRight"]      .Value;
            var handleftrecent  = Points["HandLeftRecent"] .Value;
            var handrightrecent = Points["HandRightRecent"].Value;

            DistHandLeft  = handleft.Y  - handleftrecent.Y;
            DistHandRight = handright.Y - handrightrecent.Y;
        }

        private void ComputeGuardBase()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "ElbowLeft",
                "ElbowRight",
                "SpineMid"
            }))
            {
                DistHandBaseElbow_Left     = null;
                DistHandBaseElbow_Right    = null;

                ItvElbowBaseSpineMid_Left  = null;
                ItvElbowBaseSpineMid_Right = null;

                return;
            }

            var handleft   = Points["HandLeft"]  .Value;
            var handright  = Points["HandRight"] .Value;
            var elbowleft  = Points["ElbowLeft"] .Value;
            var elbowright = Points["ElbowRight"].Value;
            var spinemid   = Points["SpineMid"]  .Value;

            DistHandBaseElbow_Left     = handleft.Y  - elbowleft.Y;
            DistHandBaseElbow_Right    = handright.Y - elbowright.Y;

            ItvElbowBaseSpineMid_Left  = Mathf.Abs(elbowleft.X  - spinemid.X);
            ItvElbowBaseSpineMid_Right = Mathf.Abs(elbowright.X - spinemid.X);
        }

        private void ComputeHandUp()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "Head"
            }))
            {
                DistHandBaseHead_Left  = null;
                DistHandBaseHead_Right = null;

                return;
            }

            var handleft  = Points["HandLeft"] .Value;
            var handright = Points["HandRight"].Value;
            var head      = Points["Head"]     .Value;

            DistHandBaseHead_Left  = handleft.Y  - head.Y;
            DistHandBaseHead_Right = handright.Y - head.Y;
        }

        private void ComputeHandDown()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "SpineMid"
            }))
            {
                DistHandBaseSpineMid_Left  = null;
                DistHandBaseSpineMid_Right = null;

                return;
            }

            var handleft  = Points["HandLeft"] .Value;
            var handright = Points["HandRight"].Value;
            var spinemid  = Points["SpineMid"] .Value;

            DistHandBaseSpineMid_Left  = handleft.Y  - spinemid.Y;
            DistHandBaseSpineMid_Right = handright.Y - spinemid.Y;
        }

        private void ComputeOnTheTable()
        {
            if (!PointsNotNull(new string[]
            {
                "KneeLeft",
                "KneeRight"
            }))
            {
                DistKneeBaseTable = null;

                return;
            }

            var kneeleft  = Points["KneeLeft"] .Value;
            var kneeright = Points["KneeRight"].Value;

            DistKneeBaseTable = Mathf.Min(kneeleft.Y, kneeright.Y) - Table;
        }

        private void ComputeJesus()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "Head"
            }))
            {
                ItvHandBaseHead = null;

                return;
            }

            var handleft  = Points["HandLeft"] .Value;
            var handright = Points["HandRight"].Value;
            var head      = Points["Head"]     .Value;

            ItvHandBaseHead = Mathf.Min(Mathf.Abs(handleft.X  - head.X),
                                        Mathf.Abs(handright.X - head.X));
        }

        private void ComputeHeadphone()
        {
            if (!PointsNotNull(new string[]
            {
                "HandLeft",
                "HandRight",
                "Head"
            }))
            {
                ItvHandBaseHead_LeftEar    = null;
                ItvHandBaseHead_RightEar   = null;
                ItvHandBaseHead_LeftDepth  = null;
                ItvHandBaseHead_RightDepth = null;

                return;
            }

            var handleft  = Points["HandLeft"] .Value;
            var handright = Points["HandRight"].Value;
            var head      = Points["Head"]     .Value;

            ItvHandBaseHead_LeftEar    = Mathf.Abs(handleft.X  - head.X);
            ItvHandBaseHead_RightEar   = Mathf.Abs(handright.X - head.X);
            ItvHandBaseHead_LeftDepth  = Mathf.Abs(handleft.Z  - head.Z);
            ItvHandBaseHead_RightDepth = Mathf.Abs(handright.Z - head.Z);
        }
    }
}