﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;

public class JudgeManager : MonoBehaviour
{
    private static JudgeManager instance;
    public static JudgeManager Instance
    {
        get
        {
            if (instance == null)
            {
                instance = FindObjectOfType(typeof(JudgeManager))
                    as JudgeManager;
            }

            return instance;
        }
    }
    
    float latency = 225f;
    float scrollMultiplier = 1.0f;

    private GameObject offset, noteobj;
    private GameObject motionGuage;
    private GameObject judgeText;

    private Vector3 initialPos;

    private float elapsedTime = 0;

    private float MsPerBeat
    {
        get
        {
            return 60 * 1000f / GameManager.Instance.CurrentTrack.BPM;
        }
    }
    
    private float ScrollSpeed
    {
        get
        {
            return scrollMultiplier * MsPerBeat / 1000f;
        }
    }

    // Use this for initialization
    void Start()
    {
        offset = GameObject.Find("Offset");
        noteobj = GameObject.Find("Noteobj");
        judgeText = GameObject.Find("Judge");
        motionGuage = GameObject.Find("Motion Guage");

        initialPos = offset.transform.position;
        judgeText.SetActive(false);
        motionGuage.transform.parent.gameObject.SetActive(false);

        LoadNotes(GameManager.Instance.CurrentTrack.Notes);
    }

    // Update is called once per frame
    void Update()
    {
        elapsedTime += Time.deltaTime * 1000;
        float timing = elapsedTime + latency;

        offset.transform.position = new Vector3(
            -timing * ScrollSpeed, 0, 0);

        if (noteobj.transform.childCount <= 0)
            return;

        GameObject obj = noteobj.transform.GetChild(0).gameObject;
        Note note = obj.GetComponent<Note.Controller>().Instance;
        
        if (note.IsLong && note.Activated)
        {
            if (InputManager.Instance.IsButtonHolding)
            {
                if(Judge.IsNonEmptyMiss(note, timing, true))
                {
                    SetJudge(Judge.MISS);
                    DeactivateNote(note);
                }
                return;
            }
            if (InputManager.Instance.IsButtonReleased)
            {
                SetJudge(Judge.TestJudge(note, timing, true));
                DeactivateNote(note);
                return;
            }
        }

        if (note is MotionNote)
        {
            ((MotionNote)note).Checkpoint();

            if (note.Activated)
            {
                SetJudge(Judge.TestJudge(note, timing));
                DeactivateNote(note);
            }

            return;
        }

        Judge judge = Judge.TestJudge(note, timing);

        if (Judge.IsNonEmptyMiss(note, timing))
        {
            SetJudge(judge);
            DeactivateNote(note);
        }

        if (InputManager.Instance.IsButtonPressed)
        {
            SetJudge(judge);
            
            if (judge == Judge.MISS) // Empty Miss
            {
                return;
            }

            if (note.IsLong)
                note.Activated = true;
            else
                DeactivateNote(note);
        }
    }

    private void DeactivateNote(Note note)
    {
        note.Activated = false;
        note.Component.transform.SetParent(offset.transform);
        note.Component.Deactivate();
    }

    private void SetJudge(Judge judge)
    {
        if (!judge.IsBreak)
            GameManager.Instance.Combo++;
        GameManager.Instance.Score += judge.Score;
        GameManager.Instance.JudgeCount[judge]++;

        Debug.Log(judge.Name);

        judgeText.SetActive(true);
        judgeText.GetComponent<Text>().text = judge.Name;
        judgeText.GetComponent<Text>().color = judge.Color;
    }

    private void LoadNotes(List<Note> notes)
    {
        foreach (Note note in notes) {
            GameObject obj = Instantiate(
                Resources.Load(note.Type.ToString(), typeof(GameObject)),
                noteobj.transform)
                as GameObject;

            if (note.IsLong)
            {
                float length = note.Length * ScrollSpeed;

                var holdTransform = obj.transform.GetChild(1)
                    .GetComponent<RectTransform>();
                var endTransform = obj.transform.GetChild(2)
                    .GetComponent<RectTransform>();

                holdTransform.SetSizeWithCurrentAnchors(
                    RectTransform.Axis.Horizontal, length);
                holdTransform.position += new Vector3(length / 2, 0, 0);
                endTransform.position += new Vector3(length, 0, 0);
            }

            obj.transform.position += initialPos + new Vector3(
                note.StartTiming * ScrollSpeed, 0, 0);

            obj.AddComponent<Note.Controller>().Instance = note;
        }
    }
}
