﻿using CustomStructure;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace TrackAnalysis
{
    public class TrackInfo
    {
        public string Title { get; private set; }
        public string Artist { get; private set; }
        public string Genre { get; private set; }
        public float BPM { get; private set; }
        public int Level { get; private set; }

        public float BeatInterval
        { get { return 4 * 60 * 1000f / BPM; } }

        public string BGM { get; private set; }

        public List<string> TrackList { get; private set; }
        
        private int locDic, locKey, locList;
        private List<KeyValuePair<string, List<string>>> NoteDatas;
        private MultiDictionary<string, string> NoteData;
        private NoteMaker maker = new NoteMaker();
        private bool isExtracted;

        public
            TrackInfo(
                string path)
            : this(
                new FileInfo(
                    path))
        { }

        public
            TrackInfo(
                FileInfo file)
        {
            MultiDictionary<string, string> parseResult = ParseBPE(file);

            ExtractTrackHeader(parseResult);

            NoteData = parseResult;
        }

        private
            MultiDictionary<string, string>
            ParseBPE(
                FileInfo file)
        {
            MultiDictionary<string, string> result = new MultiDictionary<string, string>();

            using (StreamReader reader
                = new StreamReader(new BufferedStream(file.OpenRead())))
            {
                string line;
                while ((line = reader.ReadLine()) != null)
                {
                    if (line == "" || line[0] != '#')
                        continue;

                    string[] token = line.Split(new char[] { ' ' }, 2);

                    result[token[0]] = new List<string> { token[1].Trim() };
                }
            }

            return result;
        }

        private
            void
            ExtractTrackHeader(
                MultiDictionary<string, string> parseResult)
        {
            TrackHeader header = new TrackHeader();
            header.ExtractHeader(parseResult);

            Title = header.Title;
            Artist = header.Artist;
            Genre = header.Genre;
            BPM = header.BPM;
            Level = header.Level;
            TrackList = header.TrackList;
            BGM = header.BGM;
        }

        public Note LastNote()
        {
            for (MoveNextLocation(); locDic != NoteDatas.Count; MoveNextLocation())
            {
                var code = NoteDatas[locDic].Value[locKey].Substring(locList * 2, 2);
                if (code == "00")
                    continue;

                var type = NoteDatas[locDic].Key.Substring(4, 3);
                var timing = (int.Parse(NoteDatas[locDic].Key.Substring(1, 3)) +
                    (float)locList / (NoteDatas[locDic].Value[locKey].Length / 2)) *
                    BeatInterval;
                var note = maker.Make(code, type, timing);
                
                if (type[0] == 'L' && note == null)
                    continue;

                return note;
            }

            return null;
        }

        private void MoveNextLocation()
        {
            if (!isExtracted)
            {
                isExtracted = true;
                NoteDatas = NoteData.ToList();
            }
            else
            {
                if (locDic == NoteDatas.Count)
                    return;

                ++locList;

                if (locList * 2 == NoteDatas[locDic].Value[locKey].Length)
                {
                    ++locKey;
                    locList = 0;

                    if (locKey == NoteDatas[locDic].Value.Count)
                    {
                        ++locDic;
                        locKey = 0;
                    }
                }
            }
        }
    }
}
