﻿using MotionAnalysis;
using System.Collections.Generic;

namespace StatusConvert
{
    public static class MotionStatusExtractor
    {
        private static readonly Dictionary<string, Dictionary<string, MotionState>> ShortStates =
            new Dictionary<string, Dictionary<string, MotionState>>
            {
                {
                    "Clap", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.CLAP_PREPARE },
                        { "done",    MotionState.CLAP_DONE    }
                    }
                },
                {
                    "Jump", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.JUMP_PREPARE },
                        { "done",    MotionState.JUMP_DONE    }
                    }
                },
                {
                    "PushUpLeft", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.HURRAY |
                                     MotionState.HAND_MOVE_UP_LEFT   },
                        { "done",    MotionState.HURRAY |
                                     MotionState.HAND_MOVE_DOWN_LEFT }
                    }
                },
                {
                    "PushUpRight", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.HURRAY |
                                     MotionState.HAND_MOVE_UP_RIGHT   },
                        { "done",    MotionState.HURRAY |
                                     MotionState.HAND_MOVE_DOWN_RIGHT }
                    }
                },
                {
                    "JumpPushUpLeft", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.JUMP_PREPARE |
                                     MotionState.HURRAY       |
                                     MotionState.HAND_MOVE_UP_LEFT   },
                        { "done",    MotionState.JUMP_DONE |
                                     MotionState.HURRAY    |
                                     MotionState.HAND_MOVE_DOWN_LEFT }
                    }
                },
                {
                    "JumpPushUpRight", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.JUMP_PREPARE |
                                     MotionState.HURRAY       |
                                     MotionState.HAND_MOVE_UP_RIGHT   },
                        { "done",    MotionState.JUMP_DONE |
                                     MotionState.HURRAY    |
                                     MotionState.HAND_MOVE_DOWN_RIGHT }
                    }
                },
                {
                    "GuardLeft", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.GUARD_BASE_LEFT |
                                     MotionState.HAND_MOVE_UP_LEFT   },
                        { "done",    MotionState.GUARD_BASE_LEFT |
                                     MotionState.HAND_MOVE_DOWN_LEFT }
                    }
                },
                {
                    "GuardRight", new Dictionary<string, MotionState>
                    {
                        { "prepare", MotionState.GUARD_BASE_RIGHT |
                                     MotionState.HAND_MOVE_UP_RIGHT   },
                        { "done",    MotionState.GUARD_BASE_RIGHT |
                                     MotionState.HAND_MOVE_DOWN_RIGHT }
                    }
                }
            };

        private static readonly Dictionary<string, MotionState> LongStates
            = new Dictionary<string, MotionState>
            {
                { "HandUpLeft",       MotionState.HAND_UP_LEFT    },
                { "HandUpRight",      MotionState.HAND_UP_RIGHT   },
                { "HandDownLeft",     MotionState.HAND_DOWN_LEFT  },
                { "HandDownRight",    MotionState.HAND_DOWN_RIGHT },
                { "Jesus",            MotionState.JESUS           },
                { "HeadphoneLeft",    MotionState.HEADPHONE_LEFT  },
                { "HeadphoneRight",   MotionState.HEADPHONE_RIGHT },
                { "OnTheTable",       MotionState.ON_THE_TABLE    }
            };

        private static bool IsContain(MotionState target, MotionState motion)
        {
            return (target & motion) == motion;
        }

        private  static MotionStatus ExtractShort(string name, MotionState state)
        {
            if (IsContain(state, ShortStates[name]["prepare"]))
                return MotionStatus.Prepared;
            if (IsContain(state, ShortStates[name]["done"]))
                return MotionStatus.Done;

            return MotionStatus.None;
        }

        private  static MotionStatus ExtractLong(string name, MotionState state)
        {
            if (IsContain(state, LongStates[name]))
                return MotionStatus.Prepared |
                       MotionStatus.Done;

            return MotionStatus.None;
        }

        public static MotionStatus Extract(string name, MotionState state)
        {
            if (ShortStates.ContainsKey(name))
                return ExtractShort(name, state);
            if (LongStates .ContainsKey(name))
                return ExtractLong (name, state);

            return MotionStatus.None;
        }
    }
}