﻿using UnityEngine.TestTools;
using System.Collections;
using TrackAnalysis;
using UnityEngine;
using JudgeModule;
using NUnit.Framework;
using MotionAnalysis;
using System.Collections.Generic;

class NoteConditionTests
{
    [UnityTest]
    public IEnumerator WrongInput_When_Invalid_Note_And_Input()
    {
        var counter = new WrongInputCounter(true);

        yield return counter.Count();

        var expected = true;
        var actual = counter.Cnt == counter.Total;

        Assert.AreEqual(expected, actual, "condition should be wrong input when invalid input and note");
    }

    [UnityTest]
    public IEnumerator Not_WrongInput_When_valid_Note_And_Input()
    {
        var counter = new WrongInputCounter(false);

        yield return counter.Count();

        var expected = true;
        var actual = counter.Cnt == 0;

        Assert.AreEqual(expected, actual, "condition should not be wrong input when valid input and note");
    }

    [UnityTest]
    public IEnumerator Short_Note_Entered_When_Button_Note_And_Short_Button_Input_Entered()
    {
        var notemaker = new NoteMaker();
        var note = notemaker.Make("AA", "SBT", 0);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        var expected = true;
        var actual = condition.IsShortNoteEntered(note);

        Assert.AreEqual(expected, actual, "condition should be short note entered when button note and short button input entered");
    }

    [UnityTest]
    public IEnumerator Short_Note_Entered_When_Motion_Note_And_Short_Motion_Input_Entered()
    {
        var notemaker = new NoteMaker();
        var note = notemaker.Make("CP", "SMO", 0);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.CLAP_PREPARE;

        yield return null;

        input.CurrentMotionState = MotionState.CLAP_DONE;

        var expected = true;
        var actual = condition.IsShortNoteEntered(note);

        Assert.AreEqual(expected, actual, "condition should be short note entered when motion note and long motion input entered");
    }

    [UnityTest]
    public IEnumerator Long_Note_Start_Correctly_When_Button_Note_And_Short_Button_Input_Entered()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("AA", "LBT", 0);
        var note = notemaker.Make("AA", "LBT", 1);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        var expected = true;
        var actual = condition.IsLongNoteStartCorrectly(note);

        Assert.AreEqual(expected, actual, "condition should be long note start correctly when button note and short button input entered");
    }

    [UnityTest]
    public IEnumerator Long_Note_Start_Correctly_When_Motion_Note_And_Short_Motion_Input_Entered()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("JS", "LMO", 0);
        var note = notemaker.Make("JS", "LMO", 1);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.UNKNOWN;

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        var expected = true;
        var actual = condition.IsLongNoteStartCorrectly(note);

        Assert.AreEqual(expected, actual, "condition should be long note start correctly when motion note and long motion input entered");
    }

    [UnityTest]
    public IEnumerator Long_Note_Hold_Correctly_When_Button_Note_And_Short_Button_Input_Holding()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("AA", "LBT", 0);
        var note = notemaker.Make("AA", "LBT", 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = true;

        yield return null;

        input.IsButtonDown = true;

        var expected = true;
        var actual = condition.IsLongNoteHoldCorrectly(note);

        Assert.AreEqual(expected, actual, "condition should be long note hold correctly when button note and short button input holding");
    }

    [UnityTest]
    public IEnumerator Long_Note_Hold_Correctly_When_Motion_Note_And_Short_Motion_Input_Holding()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("JS", "LMO", 0);
        var note = notemaker.Make("JS", "LMO", 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        var expected = true;
        var actual = condition.IsLongNoteHoldCorrectly(note);

        Assert.AreEqual(expected, actual, "condition should be long note hold correctly when motion note and long motion input holding");
    }

    [UnityTest]
    public IEnumerator Long_Note_Finish_Correctly_When_Short_Button_Note_And_Short_Button_Input_Stopped()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("AA", "LBT", 0);
        var note = notemaker.Make("AA", "LBT", 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = true;

        yield return null;

        input.IsButtonDown = false;

        var expected = true;
        var actual = condition.IsLongNoteFinishCorrectly(note, note.EndTiming);

        Assert.AreEqual(expected, actual, "condition should be long note finish correctly when short button note and short button input stopped");
    }

    [UnityTest]
    public IEnumerator Long_Note_Hold_Correctly_When_Short_Motion_Note_And_Short_Motion_Input_Stopped()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("JS", "LMO", 0);
        var note = notemaker.Make("JS", "LMO", 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        var expected = true;
        var actual = condition.IsLongNoteHoldCorrectly(note);

        Assert.AreEqual(expected, actual, "condition should be long note hold correctly when short motion note and long motion input holding");
    }

    [UnityTest]
    public IEnumerator Long_Note_Finish_InCorrectly_When_Button_Note_And_Short_Button_Input_Stopped_And_Not_Activated()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("AA", "LBT", 0);
        var note = notemaker.Make("AA", "LBT", 1);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = true;

        yield return null;

        input.IsButtonDown = false;

        var expected = true;
        var actual = condition.IsLongNoteFinishIncorrectly(note, note.EndTiming);

        Assert.AreEqual(expected, actual, "condition should be long note finish incorrectly when button note and short button input stopped and not activated");
    }

    [UnityTest]
    public IEnumerator Long_Note_Finish_InCorrectly_When_Motion_Note_And_Short_Motion_Input_Stopped_And_Not_Activated()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("JS", "LMO", 0);
        var note = notemaker.Make("JS", "LMO", 1);

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        yield return null;

        input.CurrentMotionState = MotionState.UNKNOWN;

        var expected = true;
        var actual = condition.IsLongNoteFinishIncorrectly(note, note.EndTiming);

        Assert.AreEqual(expected, actual, "condition should be long note finish incorrectly when motion note and long motion input stopped and not activated");
    }

    [UnityTest]
    public IEnumerator Long_Note_Finish_InCorrectly_When_Button_Note_And_Short_Button_Input_Stopped_And_Not_End()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("AA", "LBT", 0);
        var note = notemaker.Make("AA", "LBT", Judge.BAD.TimingRange(true, false) + 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.IsButtonDown = true;

        yield return null;

        input.IsButtonDown = false;

        var expected = true;
        var actual = condition.IsLongNoteFinishIncorrectly(note, note.StartTiming);

        Assert.AreEqual(expected, actual, "condition should be long note finish incorrectly when button note and short button input stopped and not end");
    }

    [UnityTest]
    public IEnumerator Long_Note_Finish_InCorrectly_When_Motion_Note_And_Short_Motion_Input_Stopped_And_Not_End()
    {
        var notemaker = new NoteMaker();
        notemaker.Make("JS", "LMO", 0);
        var note = notemaker.Make("JS", "LMO", Judge.BAD.TimingRange(true, true) + 1);
        note.Activated = true;

        var obj = new GameObject();
        var input = obj.AddComponent<InputManager>();
        var condition = new NoteCondition(input);

        yield return null;

        input.CurrentMotionState = MotionState.JESUS;

        yield return null;

        input.CurrentMotionState = MotionState.UNKNOWN;

        var expected = true;
        var actual = condition.IsLongNoteFinishIncorrectly(note, note.StartTiming);

        Assert.AreEqual(expected, actual, "condition should be long note finish incorrectly when motion note and long motion input stopped and not end");
    }

    private abstract class ConditionCounter
    {
        public abstract int Total
        { get; }

        public          int Cnt
        { get; protected set; }

        public abstract IEnumerator Count();

        protected NoteConditionPreset TestPreset(bool isbutton, bool islong, float start, float end = 0, bool activated = false)
        {
            var preset = new NoteConditionPreset();

            var notemaker = new NoteMaker();

            if (islong)
            {
                notemaker.Make(isbutton ? "AA" : "JS",
                               isbutton ? "LBT" : "LMO",
                               start);
                preset.note = notemaker.Make(isbutton ? "AA" : "JS",
                                             isbutton ? "LBT" : "LMO",
                                             end);
            }
            else
                preset.note = notemaker.Make(isbutton ? "AA" : "CP",
                                             isbutton ? "SBT" : "SMO",
                                             start);

            preset.note.Activated = activated;

            var obj = new GameObject();
            preset.input = obj.AddComponent<InputManager>();
            preset.condition = new NoteCondition(preset.input);

            return preset;
        }

        protected void TestInput(InputManager manager, InputGenerator input, bool isButton)
        {
            if (isButton)
                manager.IsButtonDown = input.Button;
            else
                manager.CurrentMotionState = input.Motion;
        }

        protected class NoteConditionPreset
        {
            public Note note;
            public InputManager input;
            public NoteCondition condition;
        }

        protected class InputGenerator
        {
            private bool IsFirst;
            private bool IsLong;
            private string Status;

            public bool Button
            {
                get
                {
                    if (IsFirst)
                        return Status == "Entered" ||
                               Status == "Continuing";

                    IsFirst = true;

                    return Status == "Stopped" ||
                           Status == "Continuing";
                }
            }

            public MotionState Motion
            {
                get
                {
                    if (IsFirst)
                    {
                        if (IsLong)
                            return Status == "Entered" ||
                                   Status == "Continuing" ?
                                   MotionState.JESUS :
                                   MotionState.UNKNOWN;

                        return Status == "Entered" ?
                                   MotionState.CLAP_DONE :
                                   MotionState.UNKNOWN;
                    }

                    IsFirst = true;

                    if (IsLong)
                        return Status == "Stopped" ||
                               Status == "Continuing" ?
                               MotionState.JESUS :
                               MotionState.UNKNOWN;

                    return MotionState.CLAP_PREPARE;
                }
            }

            public InputGenerator(string status, bool islong)
            {
                Status = status;
                IsLong = islong;
            }
        }
    }

    private class WrongInputCounter : ConditionCounter
    {
        private const int  total = 8;
        private       bool IsWrong;

        public override int Total
        {
            get
            {
                return total;
            }
        }

        public WrongInputCounter(bool isWrong)
        {
            IsWrong = isWrong;
        }

        public override IEnumerator Count()
        {
            var list = new List<bool> { true, false };
            foreach (var isButton in list)
                foreach (var isLongNote in list)
                    foreach (var isLongInput in list)
                    {
                        var preset = TestPreset(isButton,
                                                isLongNote,
                                                0,
                                                isLongNote ? 1 : 0);
                        var input = new InputGenerator(isLongInput ? "Continuing" : "Entered",
                                                       isLongInput);

                        yield return null;

                        TestInput(preset.input,
                                  input,
                                  IsWrong ? !isButton : isButton);

                        yield return null;

                        TestInput(preset.input,
                                  input,
                                  IsWrong ? !isButton : isButton);

                        if (preset.condition.IsWrongInput(preset.note))
                            ++Cnt;
                    }
        }
    }
}