﻿using NUnit.Framework;
using Windows.Kinect;
using KinectModule;
using System.Linq;
using System;
using System.Collections.Generic;

// To test the update function and the updated buffer, you need a stub for the source frame class.

public class SourceBufferTests
{
    [Test]
    public void ColorBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as byte[];
        var actual = obj.ColorBuffer;

        Assert.AreNotEqual(expected, actual, "ColorBuffer should not be null value.");
    }

    [Test]
    public void BodyIndexBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as byte[];
        var actual = obj.BodyIndexBuffer;

        Assert.AreNotEqual(expected, actual, "BodyIndexBuffer should not be null value.");
    }

    [Test]
    public void DepthBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as ushort[];
        var actual = obj.DepthBuffer;

        Assert.AreNotEqual(expected, actual, "DepthBuffer should not be null value.");
    }

    [Test]
    public void BodyBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as Body[];
        var actual = obj.BodyBuffer;

        Assert.AreNotEqual(expected, actual, "BodyBuffer should not be null value.");
    }

    [Test]
    public void ColorBuffer_Equal_Green()
    {
        var frame = new MultiSourceFrameStub(new GreenColorFrameStub(),
                                             new BodyIndexFrameDummy(),
                                             new DepthFrameDummy(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = new byte[KinectConstants.ColorWidth *
                                KinectConstants.ColorHeight *
                                KinectConstants.ColorCount];
        frame.LastColorFrame.CopyData(expected);

        var actual = obj.ColorBuffer;

        Assert.AreEqual(expected, actual, "ColorBuffer should be filled with green.");
    }

    [Test]
    public void BodyIndexBuffer_Equal_Filled_1()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new Fill1BodyIndexFrameStub(),
                                             new DepthFrameDummy(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = new byte[KinectConstants.DepthWidth *
                                KinectConstants.DepthHeight];
        frame.LastBodyIndexFrame.CopyData(expected);

        var actual = obj.BodyIndexBuffer;

        Assert.AreEqual(expected, actual, "BodyIndexBuffer should be filled with 1.");
    }

    [Test]
    public void DepthBuffer_Equal_Filled_8000()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new BodyIndexFrameDummy(),
                                             new Fill8000DepthFrameStub(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = new ushort[KinectConstants.DepthWidth *
                                  KinectConstants.DepthHeight];
        frame.LastDepthFrame.CopyData(expected);

        var actual = obj.DepthBuffer;

        Assert.AreEqual(expected, actual, "DepthBuffer should be filled with 8000.");
    }

    [Test]
    public void BodyBuffer_Equal_Clean()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new BodyIndexFrameDummy(),
                                             new DepthFrameDummy(),
                                             new CleanBodyFrameStub());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = new IBody[KinectConstants.BodyCount];
        frame.LastBodyFrame.CopyData(expected);

        var actual = obj.BodyBuffer;

        Assert.AreEqual(expected, actual, "BodyBuffer should be clean.");
    }
}

public class MultiSourceFrameStub : IMultiSourceFrame
{
    private IColorFrame _ColorFrame;
    private IBodyIndexFrame _BodyIndexFrame;
    private IDepthFrame _DepthFrame;
    private IBodyFrame _BodyFrame;

    public IColorFrame LastColorFrame
    { get { return _ColorFrame; } }
    public IBodyIndexFrame LastBodyIndexFrame
    { get { return _BodyIndexFrame; } }
    public IDepthFrame LastDepthFrame
    { get { return _DepthFrame; } }
    public IBodyFrame LastBodyFrame
    { get { return _BodyFrame; } }

    public MultiSourceFrameStub(IColorFrame color,
                                IBodyIndexFrame bodyIndex,
                                IDepthFrame depth,
                                IBodyFrame body)
    {
        _ColorFrame = color;
        _BodyIndexFrame = bodyIndex;
        _DepthFrame = depth;
        _BodyFrame = body;
    }
}

public class GreenColorFrameStub : IColorFrame
{
    private static uint[] data =
        Enumerable.Repeat<uint>(0x00FF00FF,
                                KinectConstants.ColorWidth *
                                KinectConstants.ColorHeight).ToArray();

    public void CopyData(byte[] buffer)
    {
        Buffer.BlockCopy(buffer, 0, data, 0, buffer.Length);
    }
}

public class Fill1BodyIndexFrameStub : IBodyIndexFrame
{
    public void CopyData(byte[] buffer)
    {
        for (int i = 0; i < buffer.Length; ++i)
            buffer[i] = 1;
    }
}

public class Fill8000DepthFrameStub : IDepthFrame
{
    public void CopyData(ushort[] buffer)
    {
        for (int i = 0; i < buffer.Length; ++i)
            buffer[i] = 8000;
    }
}

public class CleanBodyFrameStub : IBodyFrame
{
    public void CopyData(IBody[] buffer)
    {
        for (int i = 0; i < buffer.Length; ++i)
        {
            var tmp = new BodyFake();
            var joints = tmp.Joints;
            for(int j = 0; j < KinectConstants.JointCount; ++j)
                joints[joints.ElementAt(j).Key] = new JointStub();
            tmp.IsTracked = false;

            buffer[i] = tmp;
        }
    }
}

public class ColorFrameDummy : IColorFrame
{
    public void CopyData(byte[] buffer)
    { }
}

public class BodyIndexFrameDummy : IBodyIndexFrame
{
    public void CopyData(byte[] buffer)
    { }
}

public class DepthFrameDummy : IDepthFrame
{
    public void CopyData(ushort[] buffer)
    { }
}

public class BodyFrameDummy : IBodyFrame
{
    public void CopyData(IBody[] buffer)
    { }
}

public class BodyFake : IBody
{
    private bool _IsTracked = false;
    private Dictionary<JointType, IJoint> _Joints = new Dictionary<JointType, IJoint>
    {
        { JointType.AnkleLeft,     new JointStub() },
        { JointType.AnkleRight,    new JointStub() },
        { JointType.ElbowLeft,     new JointStub() },
        { JointType.ElbowRight,    new JointStub() },
        { JointType.FootLeft,      new JointStub() },
        { JointType.FootRight,     new JointStub() },
        { JointType.HandLeft,      new JointStub() },
        { JointType.HandRight,     new JointStub() },
        { JointType.HandTipLeft,   new JointStub() },
        { JointType.HandTipRight,  new JointStub() },
        { JointType.Head,          new JointStub() },
        { JointType.HipLeft,       new JointStub() },
        { JointType.HipRight,      new JointStub() },
        { JointType.KneeLeft,      new JointStub() },
        { JointType.KneeRight,     new JointStub() },
        { JointType.Neck,          new JointStub() },
        { JointType.ShoulderLeft,  new JointStub() },
        { JointType.ShoulderRight, new JointStub() },
        { JointType.SpineBase,     new JointStub() },
        { JointType.SpineMid,      new JointStub() },
        { JointType.SpineShoulder, new JointStub() },
        { JointType.ThumbLeft,     new JointStub() },
        { JointType.ThumbRight,    new JointStub() },
        { JointType.WristLeft,     new JointStub() },
        { JointType.WristRight,    new JointStub() }
    };

    public bool IsTracked
    {
        get
        {
            return _IsTracked;
        }
        set
        {
            _IsTracked = value;
        }
    }
    public Dictionary<JointType, IJoint> Joints
    {
        get
        {
            return _Joints;
        }
    }

    public override bool Equals(object obj)
    {
        return this.Equals(obj as IBody);
    }

    public bool Equals(IBody obj)
    {
        var keys = Joints.Select(x => x.Key);
        foreach (var key in keys)
        {
            if (!Joints[key].Equals(obj.Joints[key]))
                return false;
        }

        return IsTracked.Equals(obj.IsTracked);
    }

    public override int GetHashCode()
    {
        return 0;
    }
}

public class JointStub : IJoint
{
    private CameraSpacePoint _Position = new CameraSpacePoint { X = 0, Y = 0, Z = 0 };

    public CameraSpacePoint Position
    { get { return _Position; } }

    public override bool Equals(object obj)
    {
        return this.Equals(obj as IJoint);
    }

    public bool Equals(IJoint obj)
    {
        return Math.Abs(Position.X - obj.Position.X) < 0.05 &&
               Math.Abs(Position.Y - obj.Position.Y) < 0.05 &&
               Math.Abs(Position.Z - obj.Position.Z) < 0.05;
    }

    public override int GetHashCode()
    {
        return (int)Position.X + (int)Position.Y + (int)Position.Z;
    }
}