﻿using System.Collections.Generic;
using System.Linq;
using Windows.Kinect;

namespace MotionAnalysis
{
    public class MotionDiscriminator
    {
        private DistItvExtractor Extractor;

        public MotionState Motion
        { get; private set; }
        public bool        IsPreseted
        { get; private set; }
        
        public void Preset(KinectModule.IBody body)
        {
            IsPreseted = false;

            if (DoNotHaveKneeJoint(body))
                return;

            Extractor = new DistItvExtractor(ComputeKneeMean(body));

            IsPreseted = true;

            Init();
        }

        private bool DoNotHaveKneeJoint(KinectModule.IBody body)
        {
            KinectModule.IJoint left,
                                right;

            return !body.Joints.TryGetValue(JointType.KneeLeft,  out left) ||
                   !body.Joints.TryGetValue(JointType.KneeRight, out right);
        }

        private float ComputeKneeMean(KinectModule.IBody body)
        {
            return body.Joints.Where(x => x.Key == JointType.KneeLeft ||
                                          x.Key == JointType.KneeRight)
                       .Select(x => x.Value.Position.Y)
                       .Average();
        }

        private void Init()
        {
            Motion = MotionState.UNKNOWN;
        }

        public void Update(KinectModule.IBody body)
        {
            Extractor.Extract(body);

            Init();
            Determine();
        }

        void Determine()
        {
            Clap();
            Jump();
            Hurray();
            HandMove();
            GuardBase();
            HandUp();
            HandDown();
            Jesus();
            Headphone();
            OnTheTable();
        }

        void Clap()
        {
            const float distPrepare = 0.3f,
                        distDone = 0.1f;

            if      (Extractor.DistHandBaseSpineShoulder <= 0.0f)
                return;

            if      (Extractor.DistHand > distPrepare)
                Motion |= MotionState.CLAP_PREPARE;
            else if (Extractor.DistHand < distDone)
                Motion |= MotionState.CLAP_DONE;
        }

        void Jump()
        {
            const float distPrepare = 0.05f,
                        distDone = 0.0f;

            if      (Extractor.DistSpine < distPrepare)
                Motion |= MotionState.JUMP_PREPARE;
            else if (Extractor.DistSpine > distDone)
                Motion |= MotionState.JUMP_DONE;
        }

        void Hurray()
        {
            if      (Extractor.DistHandBaseHead > 0)
                Motion |= MotionState.HURRAY;
        }

        void HandMove()
        {
            if      (Extractor.DistHandLeft > 0)
                Motion |= MotionState.HAND_MOVE_UP_LEFT;
            else if (Extractor.DistHandLeft < 0)
                Motion |= MotionState.HAND_MOVE_DOWN_LEFT;

            if      (Extractor.DistHandRight > 0)
                Motion |= MotionState.HAND_MOVE_UP_RIGHT;
            else if (Extractor.DistHandRight < 0)
                Motion |= MotionState.HAND_MOVE_DOWN_RIGHT;
        }

        void GuardBase()
        {
            const float distBase = 0.5f;

            if      (Extractor.DistHandBaseElbow_Left > 0.0f &&
                     Extractor.ItvElbowBaseSpineMid_Left < distBase)
                Motion |= MotionState.GUARD_BASE_LEFT;

            if      (Extractor.DistHandBaseElbow_Right > 0.0f &&
                     Extractor.ItvElbowBaseSpineMid_Right < distBase)
                Motion |= MotionState.GUARD_BASE_RIGHT;
        }

        void HandUp()
        {
            const float distUp = 0.2f;

            if      (Extractor.DistHandBaseHead_Left > distUp)
                Motion |= MotionState.HAND_UP_LEFT;

            if      (Extractor.DistHandBaseHead_Right > distUp)
                Motion |= MotionState.HAND_UP_RIGHT;
        }

        void HandDown()
        {
            if      (Extractor.DistHandBaseSpineMid_Left < 0.0f)
                Motion |= MotionState.HAND_DOWN_LEFT;

            if      (Extractor.DistHandBaseSpineMid_Right < 0.0f)
                Motion |= MotionState.HAND_DOWN_RIGHT;
        }

        void Jesus()
        {
            const float distJesus = 0.5f;

            if      (Extractor.DistHandBaseSpineShoulder > 0.0f && Extractor.ItvHandBaseHead >= distJesus)
                Motion |= MotionState.JESUS;
        }

        void Headphone()
        {
            const float itvDepth = 0.2f,
                        itvEar = 0.2f;

            if      (Extractor.ItvHandBaseHead_LeftEar < itvEar &&
                     Extractor.ItvHandBaseHead_LeftDepth < itvDepth)
                Motion |= MotionState.HEADPHONE_LEFT;

            if      (Extractor.ItvHandBaseHead_RightEar < itvEar &&
                     Extractor.ItvHandBaseHead_RightDepth < itvDepth)
                Motion |= MotionState.HEADPHONE_RIGHT;
        }

        void OnTheTable()
        {
            const float distTable = 0.5f;

            if      (Extractor.DistKneeBaseTable >= distTable)
                Motion |= MotionState.ON_THE_TABLE;
        }
    }
}
