﻿using UnityEngine;
using Windows.Kinect;

namespace MotionAnalysis
{
    public class DistItvExtractor
    {
        private CameraSpacePoint Head,
                                 HandLeft,
                                 HandRight,
                                 SpineShoulder,
                                 SpineMid,
                                 ElbowLeft,
                                 ElbowRight,
                                 KneeLeft,
                                 KneeRight,
                                 SpineMidRecent,
                                 HandLeftRecent,
                                 HandRightRecent;
    
        private float            Table;
    
        public float DistHandBaseSpineShoulder
        { get; private set; }
        public float DistHand
        { get; private set; }
        public float DistSpine
        { get; private set; }
        public float DistHandBaseHead
        { get; private set; }
        public float DistHandLeft
        { get; private set; }
        public float DistHandRight
        { get; private set; }
        public float DistHandBaseElbow_Left
        { get; private set; }
        public float DistHandBaseElbow_Right
        { get; private set; }
        public float DistHandBaseHead_Left
        { get; private set; }
        public float DistHandBaseHead_Right
        { get; private set; }
        public float DistHandBaseSpineMid_Left
        { get; private set; }
        public float DistHandBaseSpineMid_Right
        { get; private set; }
        public float DistKneeBaseTable
        { get; private set; }
        public float ItvElbowBaseSpineMid_Left
        { get; private set; }
        public float ItvElbowBaseSpineMid_Right
        { get; private set; }
        public float ItvHandBaseHead
        { get; private set; }
        public float ItvHandBaseHead_LeftEar
        { get; private set; }
        public float ItvHandBaseHead_RightEar
        { get; private set; }
        public float ItvHandBaseHead_LeftDepth
        { get; private set; }
        public float ItvHandBaseHead_RightDepth
        { get; private set; }
    
        public DistItvExtractor(float table)
        {
            Table = table;
        }
    
        public void Extract(KinectModule.IBody body)
        {
            UpdatePosition(body);
            UpdateDistItv();
            SaveRecent();
        }
    
        private void UpdatePosition(KinectModule.IBody body)
        {
            Head          = body.Joints[JointType.Head]         .Position;
            HandLeft      = body.Joints[JointType.HandLeft]     .Position;
            HandRight     = body.Joints[JointType.HandRight]    .Position;
            SpineShoulder = body.Joints[JointType.SpineShoulder].Position;
            SpineMid      = body.Joints[JointType.SpineMid]     .Position;
            ElbowLeft     = body.Joints[JointType.ElbowLeft]    .Position;
            ElbowRight    = body.Joints[JointType.ElbowRight]   .Position;
            KneeLeft      = body.Joints[JointType.KneeLeft]     .Position;
            KneeRight     = body.Joints[JointType.KneeRight]    .Position;
        }
        
        private void SaveRecent()
        {
            SpineMidRecent  = SpineMid;
            HandLeftRecent  = HandLeft;
            HandRightRecent = HandRight;
        }
        
        private void UpdateDistItv()
        {
            ComputeClap();
            ComputeJump();
            ComputeHurray();
            ComputeHandMove();
            ComputeGuardBase();
            ComputeHandUp();
            ComputeHandDown();
            ComputeOnTheTable();
            ComputeJesus();
            ComputeHeadphone();
        }

        private void ComputeClap()
        {
            DistHandBaseSpineShoulder = Mathf.Min(HandLeft.Y, HandRight.Y) - SpineShoulder.Y;

            DistHand                  = Distance(HandLeft, HandRight);
        }

        private void ComputeJump()
        {
            DistSpine = SpineMid.Y - SpineMidRecent.Y;
        }

        private void ComputeHurray()
        {
            DistHandBaseHead = Mathf.Min(HandLeft.Y, HandRight.Y) - Head.Y;
        }

        private void ComputeHandMove()
        {
            DistHandLeft  = HandLeft.Y  - HandLeftRecent.Y;
            DistHandRight = HandRight.Y - HandRightRecent.Y;
        }

        private void ComputeGuardBase()
        {
            DistHandBaseElbow_Left     = HandLeft.Y  - ElbowLeft.Y;
            DistHandBaseElbow_Right    = HandRight.Y - ElbowRight.Y;

            ItvElbowBaseSpineMid_Left  = Mathf.Abs(ElbowLeft.X  - SpineMid.X);
            ItvElbowBaseSpineMid_Right = Mathf.Abs(ElbowRight.X - SpineMid.X);
        }

        private void ComputeHandUp()
        {
            DistHandBaseHead_Left  = HandLeft.Y  - Head.Y;
            DistHandBaseHead_Right = HandRight.Y - Head.Y;
        }

        private void ComputeHandDown()
        {
            DistHandBaseSpineMid_Left  = HandLeft.Y  - SpineMid.Y;
            DistHandBaseSpineMid_Right = HandRight.Y - SpineMid.Y;
        }

        private void ComputeOnTheTable()
        {
            DistKneeBaseTable = Mathf.Min(KneeLeft.Y, KneeRight.Y) - Table;
        }

        private void ComputeJesus()
        {
            ItvHandBaseHead = Mathf.Min(Mathf.Abs(HandLeft.X - Head.X),
                                        Mathf.Abs(HandRight.X - Head.X));
        }

        private void ComputeHeadphone()
        {
            ItvHandBaseHead_LeftEar    = Mathf.Abs(HandLeft.X  - Head.X);
            ItvHandBaseHead_RightEar   = Mathf.Abs(HandRight.X - Head.X);
            ItvHandBaseHead_LeftDepth  = Mathf.Abs(HandLeft.Z  - Head.Z);
            ItvHandBaseHead_RightDepth = Mathf.Abs(HandRight.Z - Head.Z);
        }

        private float Distance(CameraSpacePoint a, CameraSpacePoint b)
        {
            return Mathf.Sqrt(Mathf.Pow(a.X - b.X, 2.0f) +
                              Mathf.Pow(a.Y - b.Y, 2.0f) +
                              Mathf.Pow(a.Z - b.Z, 2.0f));
        }
    }
}