﻿using NUnit.Framework;
using Windows.Kinect;
using System.Linq;
using System;
using System.Collections.Generic;
using KinectModule;

public class SourceBufferTests
{
    [Test]
    public void ColorBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as byte[];
        var actual = obj.ColorBuffer;

        Assert.AreNotEqual(expected, actual, "ColorBuffer should not be null value.");
    }

    [Test]
    public void BodyIndexBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as byte[];
        var actual = obj.BodyIndexBuffer;

        Assert.AreNotEqual(expected, actual, "BodyIndexBuffer should not be null value.");
    }

    [Test]
    public void DepthBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as ushort[];
        var actual = obj.DepthBuffer;

        Assert.AreNotEqual(expected, actual, "DepthBuffer should not be null value.");
    }

    [Test]
    public void BodyBuffer_NotEqual_Null()
    {
        var obj = new SourceBuffer();

        var expected = null as Body[];
        var actual = obj.BodyBuffer;

        Assert.AreNotEqual(expected, actual, "BodyBuffer should not be null value.");
    }

    [Test]
    public void ColorBuffer_Equal_Green()
    {
        var frame = new MultiSourceFrameStub(new GreenColorFrameStub(),
                                             new BodyIndexFrameDummy(),
                                             new DepthFrameDummy(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = frame.LastColorFrame.GetData();

        var actual = obj.ColorBuffer;

        Assert.AreEqual(expected, actual, "ColorBuffer should be filled with green.");
    }

    [Test]
    public void BodyIndexBuffer_Equal_Filled_1()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new Fill1BodyIndexFrameStub(),
                                             new DepthFrameDummy(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = frame.LastBodyIndexFrame.GetData();

        var actual = obj.BodyIndexBuffer;

        Assert.AreEqual(expected, actual, "BodyIndexBuffer should be filled with 1.");
    }

    [Test]
    public void DepthBuffer_Equal_Filled_8000()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new BodyIndexFrameDummy(),
                                             new Fill8000DepthFrameStub(),
                                             new BodyFrameDummy());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = frame.LastDepthFrame.GetData();

        var actual = obj.DepthBuffer;

        Assert.AreEqual(expected, actual, "DepthBuffer should be filled with 8000.");
    }

    [Test]
    public void BodyBuffer_Equal_Clean()
    {
        var frame = new MultiSourceFrameStub(new ColorFrameDummy(),
                                             new BodyIndexFrameDummy(),
                                             new DepthFrameDummy(),
                                             new CleanBodyFrameStub());

        var obj = new SourceBuffer();
        obj.UpdateBuffers(frame);

        var expected = frame.LastBodyFrame.GetData();

        var actual = obj.BodyBuffer;

        Assert.AreEqual(expected, actual, "BodyBuffer should be clean.");
    }
}

public class MultiSourceFrameStub : IMultiSourceFrame
{
    public IColorFrame     LastColorFrame
    { get; private set; }
    public IBodyIndexFrame LastBodyIndexFrame
    { get; private set; }
    public IDepthFrame     LastDepthFrame
    { get; private set; }
    public IBodyFrame      LastBodyFrame
    { get; private set; }

    public MultiSourceFrameStub(IColorFrame color,
                                IBodyIndexFrame bodyIndex,
                                IDepthFrame depth,
                                IBodyFrame body)
    {
        LastColorFrame = color;
        LastBodyIndexFrame = bodyIndex;
        LastDepthFrame = depth;
        LastBodyFrame = body;
    }
}

public class GreenColorFrameStub : IColorFrame
{
    private static byte[] data = Enumerable.Range(0,
                                                  KinectConstants.ColorWidth *
                                                  KinectConstants.ColorHeight)
                                           .Select(x => (byte)((x & 1) == 1 ? 0xFF : 0))
                                           .ToArray();
    
    public byte[] GetData()
    {
        return data;
    }
}

public class Fill1BodyIndexFrameStub : IBodyIndexFrame
{
    private static byte[] data = Enumerable.Repeat(0,
                                                   KinectConstants.DepthWidth *
                                                   KinectConstants.DepthHeight)
                                           .Select(x => (byte)x)
                                           .ToArray();
    
    public byte[] GetData()
    {
        return data;
    }
}

public class Fill8000DepthFrameStub : IDepthFrame
{
    private static ushort[] data = Enumerable.Repeat(8000,
                                                     KinectConstants.DepthWidth *
                                                     KinectConstants.DepthHeight)
                                             .Select(x => (ushort)x)
                                             .ToArray();
    
    public ushort[] GetData()
    {
        return data;
    }
}

public class CleanBodyFrameStub : IBodyFrame
{
    private static BodyFake[] data = Enumerable.Range(0, KinectConstants.BodyCount)
                                               .Select(x =>
                                               {
                                                   var tmp = new BodyFake();
                                                   var joints = tmp.Joints;
                                                   for (int j = 0; j < KinectConstants.JointCount; ++j)
                                                       joints[joints.ElementAt(j).Key] = new JointStub();
                                                   tmp.IsTracked = false;

                                                   return tmp;
                                               })
                                               .ToArray();

    public IBody[] GetData()
    {
        return data;
    }
}

public class ColorFrameDummy : IColorFrame
{
    private static byte[] data = new byte[KinectConstants.ColorWidth *
                                          KinectConstants.ColorHeight *
                                          KinectConstants.ColorCount];

    public void CopyData(byte[] buffer)
    { }

    public byte[] GetData()
    {
        return data;
    }
}

public class BodyIndexFrameDummy : IBodyIndexFrame
{
    private static byte[] data = new byte[KinectConstants.DepthWidth *
                                          KinectConstants.DepthHeight];

    public void CopyData(byte[] buffer)
    { }

    public byte[] GetData()
    {
        return data;
    }
}

public class DepthFrameDummy : IDepthFrame
{
    private static ushort[] data = new ushort[KinectConstants.DepthWidth *
                                              KinectConstants.DepthHeight];

    public void CopyData(ushort[] buffer)
    { }

    public ushort[] GetData()
    {
        return data;
    }
}

public class BodyFrameDummy : IBodyFrame
{
    private static IBody[] data = new IBody[KinectConstants.BodyCount];

    public void CopyData(IBody[] buffer)
    { }

    public IBody[] GetData()
    {
        return data;
    }
}

public class BodyFake : IBody
{
    public bool                          IsTracked
    { get; set; }
    public Dictionary<JointType, IJoint> Joints
    { get; protected set; }

    public BodyFake()
    {
        IsTracked = false;
        Joints = new Dictionary<JointType, IJoint>
        {
            { JointType.AnkleLeft,     new JointStub() },
            { JointType.AnkleRight,    new JointStub() },
            { JointType.ElbowLeft,     new JointStub() },
            { JointType.ElbowRight,    new JointStub() },
            { JointType.FootLeft,      new JointStub() },
            { JointType.FootRight,     new JointStub() },
            { JointType.HandLeft,      new JointStub() },
            { JointType.HandRight,     new JointStub() },
            { JointType.HandTipLeft,   new JointStub() },
            { JointType.HandTipRight,  new JointStub() },
            { JointType.Head,          new JointStub() },
            { JointType.HipLeft,       new JointStub() },
            { JointType.HipRight,      new JointStub() },
            { JointType.KneeLeft,      new JointStub() },
            { JointType.KneeRight,     new JointStub() },
            { JointType.Neck,          new JointStub() },
            { JointType.ShoulderLeft,  new JointStub() },
            { JointType.ShoulderRight, new JointStub() },
            { JointType.SpineBase,     new JointStub() },
            { JointType.SpineMid,      new JointStub() },
            { JointType.SpineShoulder, new JointStub() },
            { JointType.ThumbLeft,     new JointStub() },
            { JointType.ThumbRight,    new JointStub() },
            { JointType.WristLeft,     new JointStub() },
            { JointType.WristRight,    new JointStub() }
        };
    }

    public override bool Equals(object obj)
    {
        return this.Equals(obj as IBody);
    }

    public bool Equals(IBody obj)
    {
        var keys = Joints.Select(x => x.Key);
        foreach (var key in keys)
        {
            if (!Joints[key].Equals(obj.Joints[key]))
                return false;
        }

        return IsTracked.Equals(obj.IsTracked);
    }

    public override int GetHashCode()
    {
        return 0;
    }
}

public class JointStub : IJoint
{
    public static float epsilon = 0.05f;

    public CameraSpacePoint Position
    { get; private set; }

    public JointStub(float x = 0f, float y = 0f, float z = 0f)
    {
        Position = new CameraSpacePoint { X = x, Y = y, Z = z };
    }

    public override bool Equals(object obj)
    {
        return this.Equals(obj as IJoint);
    }

    public bool Equals(IJoint obj)
    {
        return ComparePosition(Position, obj.Position);
    }

    public bool ComparePosition(CameraSpacePoint a, CameraSpacePoint b)
    {
        return CompareFloat(a.X, b.X) &&
               CompareFloat(a.Y, b.Y) &&
               CompareFloat(a.Z, b.Z);
    }

    private bool CompareFloat(float a, float b)
    {
        return Math.Abs(a - b) < epsilon;
    }

    public override int GetHashCode()
    {
        return (int)Position.X + (int)Position.Y + (int)Position.Z;
    }
}