﻿using UnityEngine.TestTools;
using System.Collections;
using UnityEngine;
using JudgeModule;
using NUnit.Framework;
using System.Collections.Generic;
using System.Linq;
using TestSupport;
using UnityEngine.UI;

public class NoteManagerTests
{
    [UnityTest]
    public IEnumerator Offset_Should_Move_In_Some_Frames()
    {
        var objects = MakeObjects();
        var manager = new NoteManager(objects, 60, () => null);

        bool result = true;

        foreach (var time in Enumerable.Range(0, 10 + 1))
        {
            yield return null;

            manager.Controll(time);

            if (IsSeriousError(objects["offset"].transform.position.x, Vector3.left.x * time, 0.001f))
                result = false;
        }

        var expected = true;
        var actual = result;

        Assert.AreEqual(expected, actual, "Offset should move in some frames");
    }

    [UnityTest]
    public IEnumerator Controllers_Should_Not_Change_Parent_When_Note_Not_Exist()
    {
        var objects = MakeObjects();
        var manager = new NoteManager(objects, 60, () => null);

        int count = objects["appear"].transform.childCount;

        yield return null;

        manager.Controll(0);

        var expected = true;
        var actual = (count == objects["appear"].transform.childCount);

        Assert.AreEqual(expected, actual, "Controllers should not change parent when notes does not exist");
    }

    [UnityTest]
    public IEnumerator Short_Note_Should_Move_Correctly()
    {
        var work = new TestWork(MakeObjects());
        yield return work.Work(
            new string[]
            {
                "SBT"
            },
            new float[,]
            {
                { 1000, 0 }
            },
            new List<TestTask>
            {
                new TestTask{ timing = 0,    name = "appear",    isSame = true,  isMemorize = true },
                new TestTask{ timing = 500,  name = "noteobj",   isSame = false, isMemorize = true },
                new TestTask{ timing = 1000, name = "noteobj",   isSame = false },
                new TestTask{ timing = 1500, name = "noteobj",   isSame = true },
                new TestTask{ timing = 2000, name = "deactives", isSame = true,  isMemorize = true },
            });

        var expected = true;
        var actual = work.isCorrect;

        Assert.AreEqual(expected, actual, "Short note should move correctly");
    }

    [UnityTest]
    public IEnumerator Long_Note_Should_Move_Correctly()
    {
        var work = new TestWork(MakeObjects());
        yield return work.Work(
            new string[]
            {
                "LBT"
            },
            new float[,]
            {
                { 1000, 1500 }
            },
            new List<TestTask>
            {
                new TestTask{ timing = 0,    name = "appear",    isSame = true,  isMemorize = true },
                new TestTask{ timing = 500,  name = "noteobj",   isSame = false, isMemorize = true },
                new TestTask{ timing = 1000, name = "noteobj",   isSame = false },
                new TestTask{ timing = 1500, name = "noteobj",   isSame = false },
                new TestTask{ timing = 1800, name = "noteobj",   isSame = true },
                new TestTask{ timing = 2000, name = "deactives", isSame = true,  isMemorize = true },
            });

        var expected = true;
        var actual = work.isCorrect;

        Assert.AreEqual(expected, actual, "Long note should move correctly");
    }

    private bool IsSeriousError(float a, float b, float errorLimit)
    {
        return Mathf.Abs(a - b) >= errorLimit;
    }

    private Dictionary<string, GameObject> MakeObjects()
    {
        var offset    = new GameObject();
        var noteobj   = new GameObject();
        var deactives = new GameObject();
        var appear    = new GameObject();
        var disappear = new GameObject();
        var judge     = new GameObject("Judge", typeof(RectTransform));

        offset   .transform.position = Vector3.zero;
        noteobj  .transform.position = Vector3.zero;
        deactives.transform.position = Vector3.zero;
        appear   .transform.position = Vector3.right * 1000;
        disappear.transform.position = Vector3.left * 1000;

        noteobj  .transform.SetParent(offset.transform);
        deactives.transform.SetParent(offset.transform);

        judge.AddComponent<Text>();

        MakeControllers(appear);

        return new Dictionary<string, GameObject>
        {
            { "offset",    offset },
            { "noteobj",   noteobj },
            { "deactives", deactives },
            { "appear",    appear },
            { "disappear", disappear },
            { "judgetext", judge }
        };
    }

    private void MakeControllers(GameObject appear)
    {
        foreach (var name in new string[]
        {
            "SBT",
            "LBT",
            "SMO",
            "LMO",
            "BeatLine",
            "MeasureLine"
        })
        {
            Enumerable.Range(0, 20).ToList().ForEach(x =>
            {
                var obj = Object.Instantiate(Resources.Load(name, typeof(GameObject)),
                    appear.transform) as GameObject;
                obj.name = name;

                var controller = obj.AddComponent<Controller>();
                controller.Instance = new Note(0, 0);
            });
        }
    }

    private class NoteGenerator
    {
        private TestNoteMaker maker = new TestNoteMaker();
        private string[] orderType;
        private float[,] orderTime;
        private int location = 0;

        public NoteGenerator(string[] type, float[,] time)
        {
            orderType = type;
            orderTime = time;
        }

        public Note GetNext()
        {
            if (location >= orderType.Length)
                return null;

            Note note = maker.MakeNote(orderType[location],
                                       orderTime[location, 0],
                                       orderTime[location, 1]);

            ++location;

            return note;
        }
    }

    private class CountMemory
    {
        private Dictionary<string, GameObject> objects;
        private Dictionary<string, int> counts = new Dictionary<string, int>();

        public CountMemory(Dictionary<string, GameObject> target)
        {
            objects = target;

            foreach (var x in objects.Keys)
                counts.Add(x, 0);
        }

        public void Memorize(string name)
        {
            counts[name] = objects[name].transform.childCount;
        }

        public bool IsSameAsMemorized(string name)
        {
            return counts[name] == objects[name].transform.childCount;
        }
    }

    private class TestTask
    {
        public float timing;
        public string name;
        public bool isMemorize;
        public bool isSame;
    }

    private class TestWork
    {
        public bool isCorrect = true;

        private Dictionary<string, GameObject> objects;

        public TestWork(Dictionary<string, GameObject> obj)
        {
            objects = obj;
        }

        public IEnumerator Work(string[] type, float[,] time, List<TestTask> tasklist)
        {
            var generator = new NoteGenerator(type, time);
            var manager = new NoteManager(objects, 60, generator.GetNext);

            var memory = new CountMemory(objects);

            foreach (var task in tasklist)
            {
                if (task.isMemorize)
                    memory.Memorize(task.name);

                yield return null;

                manager.Controll(task.timing);

                if (!(task.isSame ^ memory.IsSameAsMemorized(task.name)))
                    isCorrect = false;
            }
        }
    }
}
