﻿using NUnit.Framework;
using MotionAnalysis;
using StatusConvert;
using System.Linq;
using System.Collections.Generic;

class MotionStatusExtractorTest
{
    [Test]
    public void Output_Should_None_State_When_InvalidName()
    {
        var expected = true;
        var actual =
            StateChecker.CheckNotContain(
            "asdf",
            MotionState.ALL,
            MotionStatus.None);

        Assert.AreEqual(expected, actual, "output should be none state when entered invalid name.");
    }

    [Test]
    public void Output_Should_Prepared_State_When_Name_Clap_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckContain("Clap",
                                      MotionState.CLAP_PREPARE, 
                                      MotionStatus.Prepared);

        Assert.AreEqual(expected, actual, "output should be prepared state when name is Clap and valid input.");
    }

    [Test]
    public void Output_Should_Done_State_When_Name_Clap_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.Check("Clap", 
                               MotionState.CLAP_DONE,
                               MotionState.CLAP_PREPARE,
                               MotionStatus.Done);

        Assert.AreEqual(expected, actual, "output should be done state when name is Clap and valid input.");
    }

    [Test]
    public void Output_Should_None_State_When_Name_Clap_And_Invalid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckNotContain("Clap",
                                         MotionState.CLAP_PREPARE |
                                         MotionState.CLAP_DONE,
                                         MotionStatus.None);

        Assert.AreEqual(expected, actual, "output should be none state when name is Clap and invalid input.");
    }

    [Test]
    public void Output_Should_Prepared_State_When_Name_PushUpLeft_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckContain("PushUpLeft",
                                      MotionState.HURRAY |
                                      MotionState.HAND_MOVE_UP_LEFT,
                                      MotionStatus.Prepared);

        Assert.AreEqual(expected, actual, "output should be prepared state when name is PushUpLeft and valid input.");
    }

    [Test]
    public void Output_Should_Done_State_When_Name_PushUpLeft_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.Check("PushUpLeft",
                               MotionState.HURRAY |
                               MotionState.HAND_MOVE_DOWN_LEFT,
                               MotionState.HAND_MOVE_UP_LEFT,
                               MotionStatus.Done);

        Assert.AreEqual(expected, actual, "output should be done state when name is PushUpLeft and valid input.");
    }

    [Test]
    public void Output_Should_None_State_When_Name_PushUpLeft_And_Invalid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckNotContain("PushUpLeft",
                                         MotionState.HURRAY,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("PushUpLeft",
                                         MotionState.HURRAY |
                                         MotionState.HAND_MOVE_UP_LEFT,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("PushUpLeft",
                                         MotionState.HURRAY |
                                         MotionState.HAND_MOVE_DOWN_LEFT,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("PushUpLeft",
                                         MotionState.HAND_MOVE_UP_LEFT |
                                         MotionState.HAND_MOVE_DOWN_LEFT,
                                         MotionStatus.None);

        Assert.AreEqual(expected, actual, "output should be none state when name is PushUpLeft and invalid input.");
    }

    [Test]
    public void Output_Should_Prepared_State_When_Name_JumpPushUpLeft_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckContain("JumpPushUpLeft",
                                      MotionState.HURRAY |
                                      MotionState.JUMP_PREPARE |
                                      MotionState.HAND_MOVE_UP_LEFT,
                                      MotionStatus.Prepared);

        Assert.AreEqual(expected, actual, "output should be prepared state when name is JumpPushUpLeft and valid input.");
    }

    [Test]
    public void Output_Should_Done_State_When_Name_JumpPushUpLeft_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.Check("JumpPushUpLeft",
                               MotionState.HURRAY |
                               MotionState.JUMP_DONE |
                               MotionState.HAND_MOVE_DOWN_LEFT,
                               MotionState.JUMP_PREPARE |
                               MotionState.HAND_MOVE_UP_LEFT,
                               MotionStatus.Done);

        Assert.AreEqual(expected, actual, "output should be done state when name is JumpPushUpLeft and valid input.");
    }

    [Test]
    public void Output_Should_None_State_When_Name_JumpPushUpLeft_And_Invalid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckNotContain("JumpPushUpLeft",
                                         MotionState.HURRAY,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("JumpPushUpLeft",
                                         MotionState.HURRAY |
                                         MotionState.JUMP_PREPARE |
                                         MotionState.HAND_MOVE_UP_LEFT,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("JumpPushUpLeft",
                                         MotionState.HURRAY |
                                         MotionState.JUMP_DONE |
                                         MotionState.HAND_MOVE_DOWN_LEFT,
                                         MotionStatus.None) ||
            StateChecker.CheckNotContain("JumpPushUpLeft",
                                         MotionState.JUMP_PREPARE |
                                         MotionState.JUMP_DONE |
                                         MotionState.HAND_MOVE_UP_LEFT |
                                         MotionState.HAND_MOVE_DOWN_LEFT,
                                         MotionStatus.None);

        Assert.AreEqual(expected, actual, "output should be none state when name is JumpPushUpLeft and invalid input.");
    }

    [Test]
    public void Output_Should_Prepared_Done_State_When_Name_Jesus_And_Valid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckContain("Jesus",
                                      MotionState.JESUS,
                                      MotionStatus.Prepared |
                                      MotionStatus.Done);

        Assert.AreEqual(expected, actual, "output should be done state when name is Jesus and valid input.");
    }

    [Test]
    public void Output_Should_None_State_When_Name_Jesus_And_Invalid_Input()
    {
        var expected = true;
        var actual =
            StateChecker.CheckNotContain("Jesus",
                                         MotionState.JESUS,
                                         MotionStatus.None);

        Assert.AreEqual(expected, actual, "output should be none state when name is Jesus and invalid input.");
    }
}

public static class StateChecker
{
    public static bool Check(string name, MotionState contain, MotionState none, MotionStatus result)
    {
        return AreCorrectResults(name, result, ConditionalPossibiltiy(contain, none));
    }

    public static bool CheckContain(string name, MotionState contain, MotionStatus result)
    {
        return AreCorrectResults(name, result, AllPossibilityContainAllState(contain));
    }

    public static bool CheckNotContain(string name, MotionState none, MotionStatus result)
    {
        return AreCorrectResults(name, result, AllPossibilityNotContainAllState(none));
    }

    private static IEnumerable<MotionState> AllPossibility()
    {
        return Enumerable.Range(0, (int)MotionState.ALL + 1)
                         .Select(x => (MotionState)x);
    }

    private static IEnumerable<MotionState> AllPossibilityContainAllState(MotionState contain)
    {
        return AllPossibility().Where(x => (x & contain) == contain);
    }

    private static IEnumerable<MotionState> AllPossibilityNotContainAllState(MotionState none)
    {
        return AllPossibility().Where(x => (x & none) == MotionState.UNKNOWN);
    }

    private static IEnumerable<MotionState> ConditionalPossibiltiy(MotionState contain, MotionState none)
    {
        return AllPossibility().Where(x => (x & contain) == contain &&
                                           (x & none) == MotionState.UNKNOWN);
    }

    private static bool AreCorrectResults(string name, MotionStatus result, IEnumerable<MotionState> possibilities)
    {
        return NumberCorrectResults(name, result, possibilities) == possibilities.Count();
    }

    private static int NumberCorrectResults(string name, MotionStatus result, IEnumerable<MotionState> possibilities)
    {
        return possibilities.Select(x => MotionStatusExtractor.Extract(name, x))
                            .Where(x => x == result)
                            .Count();
    }
}
