﻿using MotionAnalysis;
using NUnit.Framework;
using System.Collections.Generic;
using Windows.Kinect;

public class MotionDiscriminatorTests
{
    [Test]
    public void IsPreseted_False_When_Constructed()
    {
        var obj = new MotionDiscriminator();

        var expected = false;
        var actual = obj.IsPreseted;

        Assert.AreEqual(expected, actual, "IsPreseted should be false when constructed");
    }

    [Test]
    public void Motion_Unknown_When_Constructed()
    {
        var obj = new MotionDiscriminator();

        var expected = MotionState.UNKNOWN;
        var actual = obj.Motion;

        Assert.AreEqual(expected, actual, "Motion should be unknown when constructed");
    }

    [Test]
    public void IsPreset_False_When_Preset_Without_KneeJoints()
    {
        var obj = new MotionDiscriminator();
        obj.Preset(new BodyDummy());

        var expected = false;
        var actual = obj.IsPreseted;

        Assert.AreEqual(expected, actual, "IsPreseted should be false when preset without knee joints");
    }

    [Test]
    public void IsPreset_True_When_Preset_With_KneeJoints()
    {
        var obj = new MotionDiscriminator();
        obj.Preset(new KneeJointsBasicBodyStub());

        var expected = true;
        var actual = obj.IsPreseted;

        Assert.AreEqual(expected, actual, "IsPreseted should be true when preset with knee joints");
    }

    [Test]
    public void Motion_Unknown_When_DistHandBaseSpineShoulder_Smaller_Than_0()
    {
        var obj = new MotionDiscriminator();
        obj.Preset(new KneeJointsBasicBodyStub());
        obj.Update(new ClapBodyStub_Unknown());

        var expected = MotionState.UNKNOWN;
        var actual = obj.Motion;

        Assert.AreEqual(expected, actual, "Motion should be unknown when DistHandBaseSpineShoulder smaller than 0");
    }

    [Test]
    public void Motion_ClapPrepare_When_ItvHand_Bigger_Than_itvPrepare()
    {
        var obj = new MotionDiscriminator();
        obj.Preset(new KneeJointsBasicBodyStub());
        obj.Update(new ClapBodyStub_Prepare());

        var expected = MotionState.CLAP_PREPARE;
        var actual = obj.Motion;

        Assert.AreEqual(expected, actual, "Motion should be clap-prepare when ItvHand bigger than itvPrepare");
    }

    [Test]
    public void Motion_ClapDone_When_ItvHand_Smaller_Than_itvDone()
    {
        var obj = new MotionDiscriminator();
        obj.Preset(new KneeJointsBasicBodyStub());
        obj.Update(new ClapBodyStub_Done());

        var expected = MotionState.CLAP_DONE;
        var actual = obj.Motion;

        Assert.AreEqual(expected, actual, "Motion should be clap-done when ItvHand smaller than itvDone");
    }
}

public class BodyDummy : BodyFake
{
    public BodyDummy()
    {
        Joints = new Dictionary<JointType, KinectModule.IJoint>();
    }
}

public class KneeJointsBasicBodyStub : BodyFake
{
    public KneeJointsBasicBodyStub()
    {
        Joints = new Dictionary<JointType, KinectModule.IJoint>
        {
            { JointType.KneeLeft,  new JointStub() },
            { JointType.KneeRight, new JointStub() }
        };
    }
}

public class ClapBodyStub_Unknown : BodyFake
{
    public ClapBodyStub_Unknown()
    {
        Joints = new Dictionary<JointType, KinectModule.IJoint>
        {
            { JointType.HandLeft,      new JointStub() },
            { JointType.HandRight,     new JointStub() },
            { JointType.SpineShoulder, new JointStub(y: 1) }
        };
    }
}

public class ClapBodyStub_Prepare : BodyFake
{
    public ClapBodyStub_Prepare()
    {
        Joints = new Dictionary<JointType, KinectModule.IJoint>
        {
            { JointType.HandLeft,      new JointStub() },
            { JointType.HandRight,     new JointStub(x: MotionDiscriminator.itvPrepare + 1) },
            { JointType.SpineShoulder, new JointStub(y: -1) }
        };
    }
}

public class ClapBodyStub_Done : BodyFake
{
    public ClapBodyStub_Done()
    {
        Joints = new Dictionary<JointType, KinectModule.IJoint>
        {
            { JointType.HandLeft,      new JointStub() },
            { JointType.HandRight,     new JointStub(x: MotionDiscriminator.itvDone / 2) },
            { JointType.SpineShoulder, new JointStub(y: -1) }
        };
    }
}