﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace JudgeModule
{ 
    public class NoteManager
    {
        private GameObject offset,
                           noteobj,
                           deactives,
                           appear,
                           disappear,
                           judgetext;
    
        private Vector3 initialPos;
        private float BPM;
        private Func<Note> GetLastNote;

        private int cntWait = 20;
        private const int cntTotal = 20;
        private Dictionary<string, List<Controller>> waits
            = new Dictionary<string, List<Controller>>(),
                                                     enables
            = new Dictionary<string, List<Controller>>(),
                                                     disables
            = new Dictionary<string, List<Controller>>();
        private string[] notenames
            = new string[] { "SBT", "LBT", "SMO", "LMO" };

        public Note Current
        {
            get
            {
                float timing = 0;
                string type = null;
                foreach (var x in notenames)
                {
                    if (enables[x].Count == 0)
                        continue;

                    if (type == null || timing > enables[x][0].Instance.StartTiming)
                    {
                        type = x;
                        timing = enables[x][0].Instance.StartTiming;
                    }
                }

                if (type == null)
                    return null;
                return enables[type][0].Instance;
            }
        }
    
        private float MsPerBeat
        {
            get
            {
                return 60 * 1000f / BPM;
            }
        }
    
        private float ScrollSpeed
        {
            get
            {
                return MsPerBeat / 1000f;
            }
        }

        public bool IsEnd
        { get; private set; }
        
        public NoteManager(Dictionary<string, GameObject> gameObjects, float bpm, Func<Note> getlastnote)
        {
            offset    = gameObjects["offset"];
            noteobj   = gameObjects["noteobj"];
            deactives = gameObjects["deactives"];
            appear    = gameObjects["appear"];
            disappear = gameObjects["disappear"];
            judgetext = gameObjects["judgetext"];

            initialPos = offset.transform.position;
            BPM = bpm;
            GetLastNote = getlastnote;

            foreach (var name in new string[]
            { "SBT", "LBT", "SMO", "LMO", "BeatLine", "MeasureLine" })
            {
                waits[name] = appear.transform.Cast<Transform>()
                                    .Where(x => x.gameObject.name == name)
                                    .Select(x => x.GetComponent<Controller>())
                                    .ToList();
                enables[name] = new List<Controller>();
                disables[name] = new List<Controller>();

                foreach (var c in waits[name])
                {
                    c.enables = enables[name];
                    c.disables = disables[name];
                }
            }

            IsEnd = false;
        }
    
        public void Controll(float timing)
        {
            offset.transform.position = Vector3.left * timing * ScrollSpeed;
    
            RelocateControllers(timing);
        }
    
        private void RelocateControllers(float timing)
        {
            DeactiveUndecided();
            SwitchStandBy();
            AssignNotes(timing);
        }
    
        private void DeactiveUndecided()
        {
            foreach (var name in notenames)
            {
                var controllers = enables[name].ToArray();
                foreach (var c in controllers)
                    if (initialPos.x - c.EndPosition(ScrollSpeed) > Judge.BAD.TimingRange(c.Instance))
                    {
                        NoteJudger.SetJudge(Judge.MISS, judgetext);
                        c.Deactivate(deactives.transform);
                    }
            }
        }
    
        private void SwitchStandBy()
        {
            foreach (var name in notenames)
            {
                var controllers = disables[name].ToArray();
                foreach (var c in controllers)
                    if (c.EndPosition(ScrollSpeed) <= disappear.transform.position.x)
                    {
                        c.transform.position = appear.transform.position;
                        c.transform.SetParent(appear.transform);
                        c.gameObject.SetActive(false);
                        disables[name].Remove(c);
                        waits[name].Add(c);
                        ++cntWait;
                    }
            }
        }
    
        private void AssignNotes(float timing)
        {
            if (cntWait == 0)
                return;

            var notes = new List<Note>();
            for (int i = 0; i < cntWait; ++i)
            {
                Note note = GetLastNote();
                if (note == null)
                    break;
                notes.Add(note);
            }

            cntWait -= notes.Count;
            if (cntWait == cntTotal)
            {
                IsEnd = true;
                return;
            }
            
            foreach (var name in notenames)
                AssignSpecificNotes(timing, notes, name);
        }

        private void AssignSpecificNotes(float timing, List<Note> lastnotes, string name)
        {
            var notes = GetSpecificNotes(lastnotes,
                (NoteType)Enum.Parse(typeof(NoteType), name));

            var cntActive = notes.Count < waits[name].Count ?
                notes.Count : waits[name].Count;

            enables[name].AddRange(waits[name].GetRange(0, cntActive));
            waits[name].RemoveRange(0, cntActive);

            for (int i = 0; i < cntActive; ++i)
                AssignNote(notes[i], enables[name][i], timing);
        }

        private List<Note> GetSpecificNotes(IEnumerable<Note> lastnotes, NoteType type)
        {
            return lastnotes.Where(x => x.Type == type)
                            .ToList();
        }

        private void AssignNote(Note note, Controller controller, float timing)
        {
            controller.Instance = note;
            note.Component = controller;

            controller.transform.position = initialPos +
            Vector3.right * ((note.StartTiming - timing) * ScrollSpeed);

            controller.transform.SetParent(noteobj.transform);
            controller.gameObject.SetActive(true);

            if (note.IsLong)
                StretchLongNote(controller);
        }

        private void StretchLongNote(Controller controller)
        {
            float length = controller.Instance.Length * ScrollSpeed;

            var hold = controller.transform.GetChild(1).GetComponent<RectTransform>();
            var end  = controller.transform.GetChild(2).GetComponent<RectTransform>();

            hold.SetSizeWithCurrentAnchors(RectTransform.Axis.Horizontal, length);

            hold.Translate(Vector3.right * (length / 2));
            end .Translate(Vector3.right * length);
        }
    }
}