﻿using KinectModule;
using System.Linq;
using Windows.Kinect;

namespace MotionAnalysis
{
    public class MotionDiscriminator
    {
        private DistItvExtractor Extractor;

        public MotionState Motion
        { get; private set; }
        public bool        IsPreseted
        { get; private set; }

        public const float itvPrepare  = 0.3f,
                           itvDone     = 0.1f,
                           distPrepare = 0.05f,
                           distDone    = 0.0f,
                           distBase    = 0.5f,
                           distUp      = 0.2f,
                           distJesus   = 0.5f,
                           itvDepth    = 0.2f,
                           itvEar      = 0.2f,
                           distTable   = 0.5f;

        public void Preset(IBody body)
        {
            IsPreseted = false;

            if (HaveKneeJoint(body))
            {
                Extractor = new DistItvExtractor(ComputeKneeMean(body));

                IsPreseted = true;

                Init();
            }
        }

        private bool HaveKneeJoint(IBody body)
        {
            return body        != null                         &&
                   body.Joints != null                         &&
                   body.Joints.ContainsKey(JointType.KneeLeft) &&
                   body.Joints.ContainsKey(JointType.KneeRight);
        }

        private float ComputeKneeMean(IBody body)
        {
            return body.Joints.Where(x => x.Key == JointType.KneeLeft ||
                                          x.Key == JointType.KneeRight)
                       .Select(x => x.Value.Position.Y)
                       .Average();
        }

        private void Init()
        {
            Motion = MotionState.UNKNOWN;
        }

        public void Update(IBody body)
        {
            Extractor.Extract(body);

            Init();
            Determine();
        }

        private void Determine()
        {
            Clap();
            Jump();
            Hurray();
            HandMove();
            GuardBase();
            HandUp();
            HandDown();
            Jesus();
            Headphone();
            OnTheTable();
        }

        private void Clap()
        {
            if      (Extractor.DistHandBaseSpineShoulder <= 0.0f)
                return;

            if      (Extractor.ItvHand > itvPrepare)
                Motion |= MotionState.CLAP_PREPARE;
            else if (Extractor.ItvHand < itvDone)
                Motion |= MotionState.CLAP_DONE;
        }

        private void Jump()
        {
            if      (Extractor.DistSpine < distPrepare)
                Motion |= MotionState.JUMP_PREPARE;
            else if (Extractor.DistSpine > distDone)
                Motion |= MotionState.JUMP_DONE;
        }

        private void Hurray()
        {
            if      (Extractor.DistHandBaseHead > 0)
                Motion |= MotionState.HURRAY;
        }

        private void HandMove()
        {
            if      (Extractor.DistHandLeft > 0)
                Motion |= MotionState.HAND_MOVE_UP_LEFT;
            else if (Extractor.DistHandLeft < 0)
                Motion |= MotionState.HAND_MOVE_DOWN_LEFT;

            if      (Extractor.DistHandRight > 0)
                Motion |= MotionState.HAND_MOVE_UP_RIGHT;
            else if (Extractor.DistHandRight < 0)
                Motion |= MotionState.HAND_MOVE_DOWN_RIGHT;
        }

        private void GuardBase()
        {
            if      (Extractor.DistHandBaseElbow_Left > 0.0f &&
                     Extractor.ItvElbowBaseSpineMid_Left < distBase)
                Motion |= MotionState.GUARD_BASE_LEFT;

            if      (Extractor.DistHandBaseElbow_Right > 0.0f &&
                     Extractor.ItvElbowBaseSpineMid_Right < distBase)
                Motion |= MotionState.GUARD_BASE_RIGHT;
        }

        private void HandUp()
        {
            if      (Extractor.DistHandBaseHead_Left > distUp)
                Motion |= MotionState.HAND_UP_LEFT;

            if      (Extractor.DistHandBaseHead_Right > distUp)
                Motion |= MotionState.HAND_UP_RIGHT;
        }

        private void HandDown()
        {
            if      (Extractor.DistHandBaseSpineMid_Left < 0.0f)
                Motion |= MotionState.HAND_DOWN_LEFT;

            if      (Extractor.DistHandBaseSpineMid_Right < 0.0f)
                Motion |= MotionState.HAND_DOWN_RIGHT;
        }

        private void Jesus()
        {
            if      (Extractor.DistHandBaseSpineShoulder > 0.0f && Extractor.ItvHandBaseHead >= distJesus)
                Motion |= MotionState.JESUS;
        }

        private void Headphone()
        {
            if      (Extractor.ItvHandBaseHead_LeftEar < itvEar &&
                     Extractor.ItvHandBaseHead_LeftDepth < itvDepth)
                Motion |= MotionState.HEADPHONE_LEFT;

            if      (Extractor.ItvHandBaseHead_RightEar < itvEar &&
                     Extractor.ItvHandBaseHead_RightDepth < itvDepth)
                Motion |= MotionState.HEADPHONE_RIGHT;
        }

        private void OnTheTable()
        {
            if      (Extractor.DistKneeBaseTable >= distTable)
                Motion |= MotionState.ON_THE_TABLE;
        }
    }
}
