﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using Windows.Kinect;
using System.IO;

public class KinectScript : MonoBehaviour
{

    KinectSensor sensor;
    MultiSourceFrameReader msfReader;

    KinectImage image;
    KinectMotion motion = new KinectMotion();

    bool isUpdating = false;

    // Use this for initialization
    void Start()
    {
        sensor = KinectSensor.GetDefault();
        if (sensor == null)
            return;

        msfReader = sensor.OpenMultiSourceFrameReader(FrameSourceTypes.Color |
                                                      FrameSourceTypes.Depth |
                                                      FrameSourceTypes.Body);

        image = new KinectImage(
            sensor.ColorFrameSource.CreateFrameDescription(
                ColorImageFormat.Rgba));

        if (!sensor.IsOpen)
            sensor.Open();
    }

    // Update is called once per frame
    void Update()
    {
        if (isUpdating)
            Update(image);
    }

    void AccurateUpdate()
    {
        if (isUpdating)
            Update(motion);
    }

    void Update(KinectData data)
    {
        if (msfReader == null)
            return;

        var frame = msfReader.AcquireLatestFrame();
        if (frame == null)
            return;

        data.Update(frame);
    }

    public void StartUpdating()
    {
        isUpdating = true;

        InvokeRepeating("AccurateUpdate", 0, 0.001f);
    }

    public void StopUpdating()
    {
        isUpdating = false;

        CancelInvoke("AccurateUpdate");
    }

    private void OnApplicationQuit()
    {
        StopUpdating();

        if (msfReader == null)
            return;
        msfReader.Dispose();
        msfReader = null;

        if (sensor == null)
            return;
        if (sensor.IsOpen)
            sensor.Close();
        sensor = null;
    }

    public Texture2D GetImage()
    {
        return image.GetImage();
    }

    public Pair<float, MotionState> GetState()
    {
        return motion.GetState();
    }
}

public abstract class KinectSource
{
    public abstract void Update(MultiSourceFrame frame);
}

public class KinectBody : KinectSource
{
    Body[] body = null;
    public Dictionary<JointType, Windows.Kinect.Joint> joints
    {
        get;
        private set;
    }

    public KinectBody()
    {
        joints = null;
    }

    public override void Update(MultiSourceFrame frame)
    {
        BodyFrame bFrame = frame.BodyFrameReference.AcquireFrame();
        if (body == null)
            body = new Body[bFrame.BodyCount];
        bFrame.GetAndRefreshBodyData(body);

        bFrame.Dispose();
        bFrame = null;

        UpdateBodyData();
    }

    private void UpdateBodyData()
    {
        int idx;
        for (idx = 0; idx < body.Length; ++idx)
            if (body[idx].IsTracked)
                break;

        if (idx >= body.Length)
            return;

        joints = body[idx].Joints;
    }
}

public class KinectColor : KinectSource
{
    public byte[] texel
    { get; private set; }

    public KinectColor(FrameDescription frameDesc)
    {
        texel = new byte[frameDesc.BytesPerPixel * frameDesc.LengthInPixels];
    }

    public override void Update(MultiSourceFrame frame)
    {
        ColorFrame cFrame = frame.ColorFrameReference.AcquireFrame();
        cFrame.CopyConvertedFrameDataToArray(texel, ColorImageFormat.Rgba);
        cFrame.Dispose();
        cFrame = null;
    }
}

[System.Flags]
public enum MotionState : uint
{
    UNKNOWN = 0,
    CLAP_PREPARE = 1,
    CLAP_DONE = 2,
    HURRAY = 4
}

public abstract class KinectData
{
    public abstract void Update(MultiSourceFrame frame);
}

public class KinectImage : KinectData
{
    Texture2D image;
    KinectColor color;

    public KinectImage(FrameDescription frameDesc)
    {
        image = new Texture2D(frameDesc.Width,
                              frameDesc.Height,
                              TextureFormat.RGBA32,
                              false);
        color = new KinectColor(frameDesc);
    }

    public override void Update(MultiSourceFrame frame)
    {
        color.Update(frame);

        image.LoadRawTextureData(color.texel);
        image.Apply();
    }

    public Texture2D GetImage()
    {
        return image;
    }
}

public class Pair<T1, T2>
{
    public T1 first;
    public T2 second;
}

public class KinectMotion : KinectData
{
    Queue<Pair<float, MotionState>> motions
    = new Queue<Pair<float, MotionState>>();
    KinectBody body = new KinectBody();
    float timeRecent = 0;

    public override void Update(MultiSourceFrame frame)
    {
        body.Update(frame);

        const float UNIT_TIME = 0.001f;
        timeRecent += UNIT_TIME;

        motions.Enqueue(new Pair<float, MotionState>
        {
            first = timeRecent,
            second = DetermineState()
        });
    }

    MotionState DetermineState()
    {
        MotionState s = MotionState.UNKNOWN;

        if (body.joints == null || body.joints.Count == 0)
            return s;

        CameraSpacePoint head = body.joints[JointType.Head].Position,
                         leftHand = body.joints[JointType.HandLeft].Position,
                         rightHand = body.joints[JointType.HandRight].Position;
        if (head.Y < leftHand.Y && head.Y < rightHand.Y)
            s |= MotionState.HURRAY;
        if (Distance(leftHand, rightHand) > 0.3f)
            s |= MotionState.CLAP_PREPARE;
        if (Distance(leftHand, rightHand) > 0.1f)
            s |= MotionState.CLAP_DONE;

        return s;
    }

    float Distance(CameraSpacePoint a, CameraSpacePoint b)
    {
        return Mathf.Sqrt(Mathf.Pow(a.X - b.X, 2.0f) +
                          Mathf.Pow(a.Y - b.Y, 2.0f) +
                          Mathf.Pow(a.Z - b.Z, 2.0f));
    }

    public Pair<float, MotionState> GetState()
    {
        return motions.Dequeue();
    }
}
