﻿using UnityEngine.TestTools;
using System.Collections;
using UnityEngine;
using JudgeModule;
using NUnit.Framework;
using System.Collections.Generic;
using System.Linq;
using TestSupport;
using UnityEngine.UI;
using MotionAnalysis;

public class JudgeManagerTests
{
    [UnityTest]
    public IEnumerator Judge_Miss_And_Note_Deactive_When_Wrong_Input_Button()
    {
        var objects = MakeObjects();
        var input = MakeInput();
        var manager = new JudgeManager(input, () => { }, objects, 60);
        var maker = new TestNoteMaker();

        var controller = objects["appear"].transform
                                          .GetChild(0)
                                          .GetComponent<Controller>();
        controller.transform.SetParent(objects["noteobj"].transform);
        controller.Instance = maker.MakeNote("SMO", 0);
        controller.Instance.Component = controller;

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        manager.JudgeNote(controller.Instance, 0);

        var expected = true;
        var actual = (objects["judgetext"].GetComponent<Text>().text == Judge.MISS.Name &&
            controller.transform.parent.name == "Deactives");

        Assert.AreEqual(expected, actual, "Judge is miss and note is deactive when wrong input with button");
    }

    [UnityTest]
    public IEnumerator Judge_Miss_And_Note_Deactive_When_Wrong_Input_Motion()
    {
        var objects = MakeObjects();
        var input = MakeInput();
        var manager = new JudgeManager(input, () => { }, objects, 60);
        var maker = new TestNoteMaker();

        var controller = objects["appear"].transform
                                          .GetChild(0)
                                          .GetComponent<Controller>();
        controller.transform.SetParent(objects["noteobj"].transform);
        controller.Instance = maker.MakeNote("SBT", 0);
        controller.Instance.Component = controller;

        yield return null;

        input.CurrentMotionState = MotionState.CLAP_PREPARE;

        yield return null;

        input.CurrentMotionState = MotionState.CLAP_DONE;

        manager.JudgeNote(controller.Instance, 0);

        var expected = true;
        var actual = (objects["judgetext"].GetComponent<Text>().text == Judge.MISS.Name &&
            controller.transform.parent.name == "Deactives");

        Assert.AreEqual(expected, actual, "Judge is miss and note is deactive when wrong input with motion");
    }

    [UnityTest]
    public IEnumerator Judge_Valid_When_Short()
    {
        var objects = MakeObjects();
        var input = MakeInput();
        var manager = new JudgeManager(input, () => { }, objects, 60);
        var maker = new TestNoteMaker();

        var controller = objects["appear"].transform
                                          .GetChild(0)
                                          .GetComponent<Controller>();

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        bool isCorrect = true;

        new Dictionary<float, Judge>
        {
            { 0,                                        Judge.JudgeList[0] },
            { Judge.JudgeList[0].ButtonTimingRange - 1, Judge.JudgeList[0] },
            { Judge.JudgeList[0].ButtonTimingRange,     Judge.JudgeList[1] },
            { Judge.JudgeList[1].ButtonTimingRange - 1, Judge.JudgeList[1] },
            { Judge.JudgeList[1].ButtonTimingRange,     Judge.JudgeList[2] },
            { Judge.JudgeList[2].ButtonTimingRange - 1, Judge.JudgeList[2] },
            { Judge.JudgeList[2].ButtonTimingRange,     Judge.JudgeList[3] }
        }
        .ToList()
        .ForEach(x =>
        {
            controller.transform.SetParent(objects["noteobj"].transform);
            controller.Instance = maker.MakeNote("SBT", 0);
            controller.Instance.Component = controller;

            manager.JudgeNote(controller.Instance, x.Key);

            if (objects["judgetext"].GetComponent<Text>().text != x.Value.Name)
                isCorrect = false;
        })
        ;

        manager.JudgeNote(controller.Instance, 0);

        var expected = true;
        var actual = isCorrect;

        Assert.AreEqual(expected, actual, "Judge is valid when short note");
    }

    [UnityTest]
    public IEnumerator Judge_Valid_When_Long()
    {
        var objects = MakeObjects();
        var input = MakeInput();
        var manager = new JudgeManager(input, () => { }, objects, 60);
        var maker = new TestNoteMaker();

        var controller = objects["appear"].transform
                                          .GetChild(0)
                                          .GetComponent<Controller>();

        bool isCorrect = true;

        controller.transform.SetParent(objects["noteobj"].transform);
        controller.Instance = maker.MakeNote("LBT", 0, 2000);
        controller.Instance.Component = controller;

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        manager.JudgeNote(controller.Instance, 0);

        if (controller.Instance.Activated == false)
            isCorrect = false;

        yield return null;

        input.IsButtonDown = true;

        manager.JudgeNote(controller.Instance, 1000);

        if (controller.Instance.JudgeCount == 0)
            isCorrect = false;

        yield return null;

        input.IsButtonDown = false;

        manager.JudgeNote(controller.Instance, 2000);

        if (controller.Instance.JudgeCount != 1 ||
            objects["judgetext"].GetComponent<Text>().text == Judge.MISS.Name)
            isCorrect = false;

        var expected = true;
        var actual = isCorrect;

        Assert.AreEqual(expected, actual, "Judge is valid when long note");
    }

    [UnityTest]
    public IEnumerator Judge_Invalid_When_Long()
    {
        var objects = MakeObjects();
        var input = MakeInput();
        var manager = new JudgeManager(input, () => { }, objects, 60);
        var maker = new TestNoteMaker();

        var controller = objects["appear"].transform
                                          .GetChild(0)
                                          .GetComponent<Controller>();

        bool isCorrect = true;

        controller.transform.SetParent(objects["noteobj"].transform);
        controller.Instance = maker.MakeNote("LBT", 0, 2000);
        controller.Instance.Component = controller;

        yield return null;

        input.IsButtonDown = false;

        yield return null;

        input.IsButtonDown = true;

        manager.JudgeNote(controller.Instance, 0);

        if (controller.Instance.Activated == false)
            isCorrect = false;

        yield return null;

        input.IsButtonDown = true;

        manager.JudgeNote(controller.Instance, 1000);

        if (controller.Instance.JudgeCount == 0)
            isCorrect = false;

        yield return null;

        input.IsButtonDown = false;

        manager.JudgeNote(controller.Instance, 1800);

        if (controller.Instance.JudgeCount != 1 ||
            objects["judgetext"].GetComponent<Text>().text != Judge.MISS.Name)
            isCorrect = false;

        var expected = true;
        var actual = isCorrect;

        Assert.AreEqual(expected, actual, "Judge is invalid when long note");
    }

    private Dictionary<string, GameObject> MakeObjects()
    {
        var offset    = new GameObject("Offset");
        var noteobj   = new GameObject("NoteObj");
        var deactives = new GameObject("Deactives");
        var appear    = new GameObject("AppearNote");
        var disappear = new GameObject("DisappearNote");
        var judgetext = new GameObject("Judge");

        offset   .transform.position = Vector3.zero;
        noteobj  .transform.position = Vector3.zero;
        deactives.transform.position = Vector3.zero;
        appear   .transform.position = Vector3.right * 1000;
        disappear.transform.position = Vector3.left * 1000;

        noteobj  .transform.SetParent(offset.transform);
        deactives.transform.SetParent(offset.transform);

        Enumerable.Range(0, 20).ToList().ForEach(x =>
        {
            var obj = new GameObject();
            obj.transform.SetParent(appear.transform);

            var controller = obj.AddComponent<Controller>();
            controller.Instance = new Note(0, 0);
        });

        judgetext.AddComponent<Text>();

        return new Dictionary<string, GameObject>
        {
            { "offset",    offset },
            { "noteobj",   noteobj },
            { "deactives", deactives },
            { "appear",    appear },
            { "disappear", disappear },
            { "judgetext", judgetext }
        };
    }

    private InputManager MakeInput()
    {
        var obj = new GameObject();
        return obj.AddComponent<InputManager>();
    }
}