﻿using System;
using UnityEngine;
using Windows.Kinect;

namespace KinectModule
{
    public class KinectAccessManager : IDisposable
    {
        private KinectSensor           _Sensor;
        private CoordinateMapper       _CoordinateMapper;
        private MultiSourceFrameReader _SourceFrameReader;

        private bool                   _IsSensorOpen      = false;
        private bool                   _IsFrameReaderOpen = false;
        private bool                   _NullFrame         = false;

        public bool             IsAccess
        {
            get
            {
                return _IsSensorOpen && _IsFrameReaderOpen;
            }
        }
        public CoordinateMapper CoordinateMapper
        {
            get
            {
                return IsAccess ? _CoordinateMapper : null;
            }
        }
        public MultiSourceFrame SourceFrame
        {
            get
            {
                if (IsAccess)
                {
                    var val = _SourceFrameReader.AcquireLatestFrame();
                    _NullFrame = (val == null);
                    return val;
                }
                _NullFrame = true;
                return null;
            }
        }
        public bool             NullFrame
        {
            get
            {
                return _NullFrame;
            }
        }


        public void Access()
        {
            OpenSensor();
            OpenFrameReader();

            _CoordinateMapper = _Sensor.CoordinateMapper;
        }

        private void OpenSensor()
        {
            if (_IsSensorOpen)
                return;

            _Sensor = KinectSensor.GetDefault();
            if (_Sensor == null)
            {
                //TODO: Popup Modal Window
                Debug.LogError("No ready Kinect found!");
                return;
            }

            _Sensor.Open();
            if (!_Sensor.IsOpen)
            {
                //TODO: Popup Modal Window
                Debug.LogError("Kinect sensor has problem!");
                CloseSensor();
                return;
            }

            _IsSensorOpen = true;
        }

        private void OpenFrameReader()
        {
            if (_IsFrameReaderOpen)
                return;

            _SourceFrameReader = _Sensor.OpenMultiSourceFrameReader(
                FrameSourceTypes.Color |
                FrameSourceTypes.Depth |
                FrameSourceTypes.BodyIndex |
                FrameSourceTypes.Body);

            _IsFrameReaderOpen = (_SourceFrameReader != null);
        }

        public void Dispose()
        {
            CloseFrameReader();
            CloseSensor();
        }

        private void CloseSensor()
        {
            if (_IsSensorOpen)
            {
                _Sensor.Close();
                _IsSensorOpen = false;
            }
        }

        private void CloseFrameReader()
        {
            if (_IsFrameReaderOpen)
            {
                _SourceFrameReader.Dispose();
                _IsFrameReaderOpen = false;
            }
        }
    }
}