﻿using UnityEngine.TestTools;
using System.Collections;
using UnityEngine;
using JudgeModule;
using NUnit.Framework;
using UnityEngine.UI;
using System.Collections.Generic;
using TestSupport;

public class NoteJudgerTests
{
    [UnityTest]
    public IEnumerator SetJudge_Should_Change_Judge_Text()
    {
        var settings = new TestSettings("SBT", 0);

        var text = settings.judgeText;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { });

        int cnt = 0;
        foreach (var x in Judge.JudgeList)
        {
            yield return null;

            NoteJudger.SetJudge(x, settings.judge);

            if (text.text == x.Name &&
                text.color == x.Color)
                ++cnt;
        }

        var expected = Judge.JudgeList.Count;
        var actual = cnt;

        Assert.AreEqual(expected, actual, "SetJudge should change judge text's text and color");
    }

    [UnityTest]
    public IEnumerator WrongNoteProc_Should_Execute_EndGame_When_Button_Note()
    {
        var settings = new TestSettings("SBT", 0);

        var note = settings.note;

        bool isEnd = false;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { isEnd = true; });

        yield return null;

        judger.WrongNoteProc(note);

        var expected = true;
        var actual = isEnd;

        Assert.AreEqual(expected, actual, "WrongNoteProc should execute endgmae when button note");
    }

    [UnityTest]
    public IEnumerator WrongNoteProc_Should_Not_Execute_EndGame_When_Motion_Note()
    {
        var settings = new TestSettings("SMO", 0);

        var note = settings.note;

        bool isEnd = false;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { isEnd = true; });

        yield return null;

        judger.WrongNoteProc(note);

        var expected = false;
        var actual = isEnd;

        Assert.AreEqual(expected, actual, "WrongNoteProc should not execute endgmae when motion note");
    }

    [UnityTest]
    public IEnumerator EnteredNoteProc_Should_Change_Activated_True_When_Long_Note()
    {
        var settings = new TestSettings("LBT", 0, Judge.BAD.TimingRange(true, false) + 10);

        var note = settings.note;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { });

        yield return null;

        judger.EnteredNoteProc(note, note.StartTiming);

        var expected = true;
        var actual = note.Activated;

        Assert.AreEqual(expected, actual, "WrongNoteProc should change activated to true when long note");
    }

    [UnityTest]
    public IEnumerator EnteredNoteProc_Should_Not_Change_Activated_True_When_Short_Note()
    {
        var settings = new TestSettings("SBT", 0);

        var note = settings.note;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { });

        yield return null;

        judger.EnteredNoteProc(note, note.StartTiming);

        var expected = false;
        var actual = note.Activated;

        Assert.AreEqual(expected, actual, "WrongNoteProc should not change activated to true when short note");
    }

    [UnityTest]
    public IEnumerator ContinuingNoteProc_Should_Judge_Perfect_When_Next_Timeing_In_Interval()
    {
        var settings = new TestSettings("LBT", 0, 1000);

        var note = settings.note;

        var judger = new NoteJudger(new Dictionary<string, GameObject>
        {
            { "deactives", settings.deactives },
            { "judgetext", settings.judge }
        },
        () => { });

        yield return null;

        note.Activated = true;

        float interval = 60f;
        float frame = 10f;

        for (float t = note.StartTiming + frame; t < note.EndTiming; t += frame)
            judger.ContinuingNoteProc(note, t, interval);

        var expected = (int)(note.EndTiming / interval);
        var actual = note.JudgeCount;

        Assert.AreEqual(expected, actual, "ContinuingNoteProc should judge perfect when next timing is included in interval");
    }

    class TestSettings
    {
        public Note note;
        public GameObject noteobj,
                          deactives,
                          judge;
        public Text judgeText;

        private TestNoteMaker maker = new TestNoteMaker();

        public TestSettings(string type, float start, float end = 0)
        {
            CreateNoteAndController(type, start, end);
            CreateJudgeText();
        }

        private void CreateNoteAndController(string type, float start, float end)
        {
            noteobj = new GameObject("NoteObj");
            deactives = new GameObject("Deactives");

            note = maker.MakeNote(type, start, end);

            var obj = new GameObject();
            obj.transform.SetParent(noteobj.transform);

            var controller = obj.AddComponent<Controller>();

            controller.Instance = note;
            note.Component = controller;

            controller.enables = new List<Controller>();
            controller.disables = controller.enables;
        }

        private void CreateJudgeText()
        {
            judge = new GameObject("Judge", typeof(RectTransform));
            judgeText = judge.AddComponent<Text>();
        }
    }
}