﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace JudgeModule
{
    public class Judge
    {
        public static readonly List<Judge> JudgeList = new List<Judge>
        {
            new Judge("PERFECT") { ButtonTimingRange = PerfectTime,  Score = 2, Color = Color.cyan },
            new Judge("GOOD")    { ButtonTimingRange = GoodTime,     Score = 1, Color = Color.yellow },
            new Judge("BAD")     { ButtonTimingRange = BadTime,                 Color = Color.blue, IsBreak = true },
            new Judge("MISS")    { Color = Color.red, IsBreak = true }
        };
    
        private Judge()
        {
            ButtonTimingRange = 0f;
            Score = 0;
            IsBreak = false;
            Color = Color.black;
        }
    
        private Judge(string name) : this() { Name = name; }
    
        public string Name { get; private set; }
    
        private const float PerfectTime = 80f,
                            GoodTime    = 100f,
                            BadTime     = 120f;

        private float ButtonTimingRange;
        public int   Score { get; private set; }
        public bool  IsBreak { get; private set; }
        public Color Color { get; private set; }
    
        public static readonly Judge BAD     = JudgeList[2];
        public static readonly Judge MISS    = JudgeList[3];
        public static readonly Judge Perfect = JudgeList[0];

        private static readonly float LongTimingBonus = 50;
        private static readonly float MotionTimingBonus = 100;

        public float TimingRange(bool isLong, bool isMotion)
        {
            return ButtonTimingRange +
                (isLong ? LongTimingBonus : 0) +
                (isMotion ? MotionTimingBonus : 0);
        }

        public float TimingRange(Note note)
        {
            return TimingRange(note.IsLong, note is MotionNote);
        }

        public static Judge TestJudge(Note note, float elapsedTime, bool end = false)
        {
            float timing = end ? note.EndTiming : note.StartTiming;
            float difference = elapsedTime - timing;
    
            var result = JudgeList.Where(x => Mathf.Abs(difference) < x.TimingRange(note)).ToList();
    
            return result.Count == 0 ? JudgeList.Last() : result[0];
        }
    
        public static bool IsPastNote(Note note, float elapsedTime)
        {
            float timing = note.IsLong && note.Activated ? note.EndTiming : note.StartTiming;
            return elapsedTime - timing > BAD.TimingRange(note);
        }
    
        public static bool IsNoteStart(Note note, float elapsedTime)
        {
            return note is MotionNote &&
                   !note.Activated;
        }
    
        public static bool IsNoteEnd(Note note, float elapsedTime)
        {
            return Mathf.Abs(elapsedTime - note.EndTiming) <= BAD.TimingRange(note);
        }

        public static bool IsNoteProgress(Note note, float elapsedTime, float interval)
        {
            var timing = note.StartTiming + (interval * (note.JudgeCount + 1));

            return Mathf.Abs(elapsedTime - timing) < Perfect.TimingRange(note) &&
                   Mathf.Abs(elapsedTime - timing) <
                       Mathf.Abs(elapsedTime - (timing - interval));
        }
    }
}