﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;

public class IngameUIManager : SingletonBehaviour<IngameUIManager>
{
	[SerializeField]
	private GameObject targetLockedUIPrefab;
	private List<RectTransform> targetLockedUIs = new List<RectTransform>();

    [SerializeField]
    private Text heightText;
    [SerializeField]
    private Text maxHeightText;

	[SerializeField]
	private GameObject enemyHealthUIPrefab;

	[SerializeField]
	private Slider playerHealthUI;

	[SerializeField]
	private Image weaponUI;
	[SerializeField]
	private List<Sprite> weaponSprites = new List<Sprite>();

	[SerializeField]
	private Slider skillUI;
	[SerializeField]
	private Text skillCooltimeText;

	[SerializeField]
	private GameObject comboUI;
	[SerializeField]
	private Text comboText;
	[SerializeField]
	private Text damageText;
	private Coroutine disableComboRoutine;

	[SerializeField]
	private Text OutOfMapWarningUI;

	[SerializeField]
	private Text distanceText;

	public void UpdateTargetLockedUIs(Vector3[] pos)
	{
		int tmp, i = 0;

		int count = targetLockedUIs.Count;

		for (int j = 0; j < pos.Length - count; j++)
		{
			targetLockedUIs.Add(Instantiate(targetLockedUIPrefab, transform).GetComponent<RectTransform>());
		}
		//Debug.Log(targetLockedUIs.Count);
		//Debug.Log(pos.Length);
		for (; i < pos.Length; i++)
		{
			Vector2 viewportPoint = Camera.main.WorldToViewportPoint(pos[i]);
			targetLockedUIs[i].anchorMin = viewportPoint;
			targetLockedUIs[i].anchorMax = viewportPoint;
		}
		for (tmp = i; i < targetLockedUIs.Count; i++)
		{
			Destroy(targetLockedUIs[i].gameObject);
		}
		if (tmp < targetLockedUIs.Count)
		{
			targetLockedUIs.RemoveRange(tmp, targetLockedUIs.Count - tmp);
		}
	}

    public void UpdateMaxHeightText(float maxHeight)
    {
        maxHeightText.text = string.Format("{0:F1}", maxHeight) + " M";
    }

    public void UpdateHeightText(float height)
    {
        heightText.text = string.Format("{0:F1}", height) + " M";
    }

	public Slider CreateNewHealthUI()
	{
		Slider ui = Instantiate(enemyHealthUIPrefab, transform).GetComponent<Slider>();
		Destroy(ui.gameObject, 5);
		return ui;
	}

	public void UpdateEnemyHealthUI(Slider ui, int maxHealth, int curHealth)
	{
		if (ui == null)
			Debug.LogError("There's no health ui available");
		ui.value = curHealth / (float)maxHealth;
	}

	public void UpdatePlayerHealthUI(int maxHealth, int curHealth)
	{
		playerHealthUI.value = (float)curHealth / maxHealth;
	}

	public void UpdateWeaponTypeUI(WeaponType type)
	{
		weaponUI.sprite = weaponSprites[(int)type];
		skillUI.gameObject.SetActive(type == WeaponType.KNIFE);
	}

	public void UpdateSkillUI(float remain, float time)
	{
		skillUI.value = 1 - remain / time;

		if (remain <= 0)
			skillCooltimeText.text = "";
		else
			skillCooltimeText.text = string.Format("{0:F1}", remain);
	}

	public void UpdateComboUI(int damage)
	{
		if (disableComboRoutine != null)
			StopCoroutine(disableComboRoutine);
		comboUI.SetActive(true);
		comboText.text = (int.Parse(comboText.text.Split(' ')[0]) + 1).ToString() + " Combo!";
		damageText.text = (int.Parse(damageText.text) + damage).ToString();
		disableComboRoutine = StartCoroutine(DisableComboUI(5));
	
	}

	private void ResetComboUI()
	{
		comboText.text = "0 Combo!";
		damageText.text = "0";
	}

	private IEnumerator DisableComboUI(float sec)
	{
		yield return new WaitForSeconds(sec);
		ResetComboUI();
		comboUI.SetActive(false);
	}

	public void UpdateOutOfMapWarningUI(float remainTime)
	{
		if (remainTime >= 10 || remainTime <= 0)
			OutOfMapWarningUI.text = "";
		else
			OutOfMapWarningUI.text = string.Format("{0:F1}",remainTime);
	}

	public void UpdateDistanceUI(float distance)
	{
		if (distance < 0)
			distanceText.text = "";
		else
			distanceText.text = string.Format("{0:F1}", distance) + "m"; ;
	}
}
