﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEditor;
using System;
using System.Linq;
using System.IO;

#if UNITY_EDITOR
public class CharacterResourceEditorWindow : EditorWindow
{
	public static Dictionary<int, CharacterResource> characters = new Dictionary<int, CharacterResource>();

	private int characterIndex = 0;


	Rect characterEditWindowRect = new Rect(0, 0, 400, 600);
	Rect characterAdderWindowRect = new Rect(400, 0, 400, 600);

	private int newCharacterID = 0;
	private string newCharacterName = "";
	private Sprite[] newCharacterImageList = new Sprite[Enum.GetNames(typeof(CharacterExpression)).Length];
	private Vector2 newCharacterScroll;

	private Vector2 characterEditScroll;

	[MenuItem("Tools/Dialogue Character Resource Editor")]
	static void Init()
	{
		CharacterResourceEditorWindow window = (CharacterResourceEditorWindow)GetWindow(typeof(CharacterResourceEditorWindow));
		characters.Clear();
		foreach (var character in Resources.LoadAll<CharacterResource>("BS_DialogueSystem/CharacterResources"))
		{
			try
			{
				characters.Add(character.characterID, character);
			}
			catch
			{
				Debug.LogError("Character ID is not unique! Check " +
					characters[character.characterID].characterName +" and " +
					character.characterName + "!");
				throw;
			}
		}
		window.Show();
	}

	private void OnGUI()
	{
		BeginWindows();
		characterEditWindowRect = GUILayout.Window(0, characterEditWindowRect, CharacterEditWindow, "Character Editor");
		characterAdderWindowRect = GUILayout.Window(1, characterAdderWindowRect, CharacterAdderWindow, "Character Adder");
		EndWindows();
	}


	private void CharacterEditWindow(int id)
	{
		string[] names = new string[characters.Count];
		int i = 0;
		foreach (var key in characters.Keys)
		{
			names[i++] = characters[key].characterName;
		}
		characterIndex = EditorGUILayout.IntPopup("Character List", characterIndex, names, characters.Keys.ToArray());

		CharacterResource character = characters[characterIndex];

		characterEditScroll = EditorGUILayout.BeginScrollView(characterEditScroll);

		/*
		EditorGUILayout.BeginHorizontal();
		EditorGUILayout.LabelField("Character ID");
		EditorGUILayout.LabelField(character.characterID.ToString());
		//character.characterID = EditorGUILayout.IntField("Character ID", character.characterID);
		EditorGUILayout.EndHorizontal();
		character.characterName = EditorGUILayout.TextField("Character Name", character.characterName);
		*/

		EditorGUILayout.LabelField("Character Images", EditorStyles.boldLabel);
		string[] expressionNames = Enum.GetNames(typeof(CharacterExpression));

		for (i = 0; i < newCharacterImageList.Length; i++)
		{
			character.characterImageList[i] = EditorGUILayout.ObjectField(expressionNames[i].ToLower(), character.characterImageList[i], typeof(Sprite), false) as Sprite;
		}
		EditorGUILayout.EndScrollView();

		characters[characterIndex] = character;

		GUI.DragWindow();
	}

	private void CharacterAdderWindow(int id)
	{
		SaveAsNewGUI();

		newCharacterScroll = EditorGUILayout.BeginScrollView(newCharacterScroll);
		newCharacterID = EditorGUILayout.IntField("Character ID", newCharacterID);
		newCharacterName = EditorGUILayout.TextField("Character Name", newCharacterName);

		EditorGUILayout.LabelField("Character Images", EditorStyles.boldLabel);
		string[] expressionNames = Enum.GetNames(typeof(CharacterExpression));

		
		for (int i = 0; i < newCharacterImageList.Length; i++)
		{
			newCharacterImageList[i] = EditorGUILayout.ObjectField(expressionNames[i].ToLower(),newCharacterImageList[i], typeof(Sprite), false) as Sprite;
		}
		EditorGUILayout.EndScrollView();
		GUI.DragWindow();
	}

	private void SaveAsNewGUI()
	{
		GUILayout.BeginHorizontal();

		bool hasSameID = false;
		foreach (var character in Resources.LoadAll<CharacterResource>("BS_DialogueSystem/CharacterResources"))
		{
			hasSameID = character.characterID == newCharacterID;
			if (hasSameID)
				break;
		}
		GUI.enabled = newCharacterName.Length > 0 &&
			Resources.Load<CharacterResource>("BS_DialougeSystem/CharacterResources" + newCharacterName) == null &&
			!hasSameID;
		if (GUILayout.Button("Save as new", GUILayout.Width(100)))
		{
			SaveAsNew();
		}
		GUILayout.EndHorizontal();

		if (!GUI.enabled)
		{
			GUIStyle style = new GUIStyle();
			style.normal.textColor = Color.red;
			if (newCharacterName.Length <= 0)
			{
				EditorGUILayout.LabelField("Name must be longer than 0!", style);
			}
			else if (hasSameID)
			{
				EditorGUILayout.LabelField("ID is already used!", style);
			}
			else
			{
				EditorGUILayout.LabelField("The same named character already exists!", style);
			}
		}
		GUI.enabled = true;
	}

	private void SaveAsNew()
	{
		CharacterResource resource = CreateInstance<CharacterResource>();
		resource.characterID = newCharacterID;
		resource.characterName = newCharacterName;
		resource.characterImageList = newCharacterImageList;

		string dir = Application.dataPath;

		if (!Directory.Exists(dir + "/Resources/BS_DialogueSystem/CharacterResources"))
		{
			Debug.Log("Creating directory for CharacterResource assets...");
			try
			{
				Directory.CreateDirectory(dir + "/Resources/BS_DialogueSystem/CharacterResources");
			}
			catch
			{
				Debug.LogError("Creating directory Failed!");
				throw;
			}
			Debug.Log("Creating directory Success!");
		}
		try
		{
			AssetDatabase.CreateAsset(resource, "Assets/Resources/BS_DialogueSystem/CharacterResources/" + newCharacterName + ".asset");
		}
		catch
		{
			Debug.LogError("Creating" + newCharacterName + ".asset Failed!");
			throw;
		}
		Debug.Log("Creating " + newCharacterName + ".asset Success!");

		characters.Add(resource.characterID, resource);

		newCharacterID = 0;
		newCharacterName = "";
		newCharacterImageList = new Sprite[Enum.GetNames(typeof(CharacterExpression)).Length];
	}
}
#endif